unit yw;

{  Retrieve weather conditions and forecasts from
   Yahoo! Weather (https://developer.yahoo.com/weather/)
   as raw text (JSON formatted) and as TJSONData object. }

{ Weather API references:

    Current weather API
       https://developer.yahoo.com/weather/

    Weather RSS Request (for response )
       https://developer.yahoo.com/weather/documentation.html
       seems to be a problem with pressure:
          in imperial it is reported in millibars not psi
          in metric is is incorrectly reported - see FIX_PRESSURE

    Yahoo Query Language (YQL)
       https://developer.yahoo.com/yql/
       https://developer.yahoo.com/yql/guide/
       https://developer.yahoo.com/yql/guide/yql-users-guide.html
}


{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, strutils, dateutils;

type
    { Application exceptions }
  TWeatherApiErr = class(Exception);

    { Weather elements that  can be requested from OpenWeatherMap
      with free account. }
  TRequestElement = (reCurrentCondition,  // the current weather conditions
                     re10DayForecast);    // daily forecast for next 10 days

  TUnits = (uMetric, uImperial);

  TDataFormat = (dfJSON, dfXML);
  TGetProtocol = (gpHTTPS, gpHTTP);

  function CityByNameLocation(const city: string; const country: string = ''): string;
  function LatLongLocation(latitude, longitude: double): string;
  function PostalCodeLocation(const code: string; const country: string = ''): string;
  function woeidLocation(const id: string): string;

  function WeatherUrl(q: TRequestElement;
                         const location: string;
                         const protocol: TGetProtocol = gpHTTPS;
                         const dataFormat: TDataFormat = dfJSON;
                         const units: TUnits = uMetric;
                         const lang: string = ''): string;

implementation

uses
  yw_consts, fphttpclient;

{  yahoo weather url

   ywurl + weatherquery with {woeid} = '= <id>' for woeidlocation

   ywulr + weatherquery with {woeid} = woeidquery with {location}
      = 'cityname[,country]'
      = 'postalcode[,country]'
      = '(latitude,longitude)'
}
const
  ywurl = '{get}://query.yahooapis.com/v1/public/yql?q=';
  weatherquery = 'select %s from weather.forecast where woeid {woeid}';
  woeidquery = 'in (select woeid from geo.places(1) where text="{location}")';

function CityByNameLocation(const city, country: string): string;
begin
  if city = '' then
    Raise TWeatherApiErr.create(SMissingCityName);
  result := city;
  if country <> '' then
    result := result + ',' + country;
  result := AnsiReplaceStr(woeidquery, '{location}', result);
end;

function LatLongLocation(latitude, longitude: double): string;
var
  defaultDecimalSeparator: char;
begin
  defaultDecimalSeparator := DefaultFormatSettings.DecimalSeparator;
  DefaultFormatSettings.DecimalSeparator := '.';
  result := Format('(%.3f,%.3f)', [latitude, longitude]);
  DefaultFormatSettings.DecimalSeparator := defaultDecimalSeparator;
  result := AnsiReplaceStr(woeidquery, '{location}', result);
end;

function PostalCodeLocation(const code, country: string): string;
begin
  if code = '' then
    Raise TWeatherApiErr.create(SMissingPostalCode);
  result := code;
  if country <> '' then
    result := result + ','+country;
  result := AnsiReplaceStr(woeidquery, '{location}', result);
end;

function woeidLocation(const id: string): string;
begin
  if id = '' then
    Raise TWeatherApiErr.create(SMissingCityID);
  result := '=' + id;
end;


function WeatherUrl(q: TRequestElement;
                       const location: string;
                       const protocol: TGetProtocol;
                       const dataFormat: TDataFormat;
                       const units: TUnits;
                       const lang: string): string;
const
  protocolstr: array[TGetProtocol] of string = ('https', 'http');

const
  requeststr: array[TRequestElement] of string = (
  {reCurrentCondition} 'location, wind, atmosphere, astronomy, item.condition',
  {re10DayForecas}     '*');


begin
  if location = '' then
    Raise TWeatherApiErr.create(SMissingLocation);
  result := Format(weatherquery, [requeststr[q]]);
  result := AnsiReplaceStr(result, '{woeid}', location);
  if units <> uImperial then
    result := result + ' and u="c"';
  result := ywurl + EncodeURLElement(result);
  result := AnsiReplaceStr(result, '{get}', protocolstr[protocol]);
  if dataFormat = dfJSON then
    result := result + '&format=json';
  //if lang <> '' then
    //result := result + '&lang=' + lang;
  // language not supported
end;


end.


