unit wb;

{  Retrieve Weatherbit.io (https://www.weatherbit.io) current
   weather conditions, weather forecasts and usage statistics
   as raw text (JSON formatted) and as TJSONData object. }

{ Weatherbit.io API references for free accounts:

    Current weather API
       https://www.weatherbit.io/api/weather-current

    120 hour forecast API
       https://www.weatherbit.io/api/weather-forecast-48-hour

    5 day forecast API
       https://www.weatherbit.io/api/weather-forecast-5-day

    16 day forecast API
       https://www.weatherbit.io/api/weather-forecast-16-day

    Subscription limits API
       https://www.weatherbit.io/api/subscription-usage

}


{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, strutils, dateutils,
    fpjson, laz2_dom;

type
    { Application exceptions }
  TWeatherApiErr = class(Exception);

    { Weather elements that  can be requested from Weatherbit.io.
      Note: historical data APIs not supported }
  TRequestElement = (reCurrentCondition,  // the current weather conditions
                     re148HourForecast,   // forecast next 48 hours (120 for paying accounts)
                     re5DayForecast,      // forecast next 5 days, 3 hour interval
                     re16DayForecast,     // forecast next 16 days
                     reUsage);            // subscription limits

  TUnits = (uImperial, uMetric, uSI);

  TDataFormat = (dfJSON, dfXML);
  TGetProtocol = (gpHTTPS, gpHTTP);

function CityByNameLocation(const city: string; const region: string = ''; const country: string = ''): string;
function LatLongLocation(latitude, longitude: double): string;
function PostalCodeLocation(const code: string; const country: string = ''): string;
function IpLocation(useAuto: boolean; const ip: string = ''): string;
function CityIdLocation(const id: string): string;
function StationIdLocation(const id: string): string;

function WeatherUrl(q: TRequestElement;
                         const location: string;
                         const protocol: TGetProtocol = gpHTTPS;
                         const dataFormat: TDataFormat = dfJSON;
                         const units: TUnits = uMetric;
                         const lang: string = ''): string;


implementation

uses
  wb_consts;

const
  wburl = '%s://api.weatherbit.io/v2.0/%s?';

function CityByNameLocation(const city, region, country: string): string;
begin
  if city = '' then
    Raise TWeatherApiErr.create(SMissingCityName);
  result := 'city=' + city;
  if region <> '' then
    result := result + ',' + region;
  if country <> '' then
    result := result + '&country=' + country;
end;

function LatLongLocation(latitude, longitude: double): string;
{
const
  USSettings: TFormatSettings = (
    CurrencyFormat: 1;
    NegCurrFormat: 5;
    ThousandSeparator: ',';
    DecimalSeparator: '.';
    CurrencyDecimals: 2;
    DateSeparator: '-';
    TimeSeparator: ':';
    ListSeparator: ',';
    CurrencyString: '$';
    ShortDateFormat: 'd/m/y';
    LongDateFormat: 'dd" "mmmm" "yyyy';
    TimeAMString: 'AM';
    TimePMString: 'PM';
    ShortTimeFormat: 'hh:nn';
    LongTimeFormat: 'hh:nn:ss';
    ShortMonthNames: ('Jan','Feb','Mar','Apr','May','Jun',
                      'Jul','Aug','Sep','Oct','Nov','Dec');
    LongMonthNames: ('January','February','March','April','May','June',
                     'July','August','September','October','November','December');
    ShortDayNames: ('Sun','Mon','Tue','Wed','Thu','Fri','Sat');
    LongDayNames:  ('Sunday','Monday','Tuesday','Wednesday','Thursday','Friday','Saturday');
    TwoDigitYearCenturyWindow: 50;
  );
begin
  result := Format('lat=%.3f&lon=%.3f', [latitude, longitude], USSettings);
end;

or could use enFormatSettings in LocalizedForms but perhaps that dependency
is not desirable.
}
var
  defaultDecimalSeparator: char;
begin
  defaultDecimalSeparator := DefaultFormatSettings.DecimalSeparator;
  DefaultFormatSettings.DecimalSeparator := '.';
  result := Format('lat=%.3f&lon=%.3f', [latitude, longitude]);
  DefaultFormatSettings.DecimalSeparator := defaultDecimalSeparator;
end;

function PostalCodeLocation(const code, country: string): string;
begin
  if code = '' then
    Raise TWeatherApiErr.create(SMissingPostalCode);
  result := 'postal_code=' + code;
  if country <> '' then
    result := result + '&country='+country;
end;

function IpLocation(useAuto: boolean; const ip: string): string;
begin
  if useAuto then
    result := 'ip=auto'
  else begin
    if ip = '' then
      Raise TWeatherApiErr.create(SMissingIpAddress);
    result := 'ip=' + ip;
  end;
end;

function CityIdLocation(const id: string): string;
begin
  if id = '' then
    Raise TWeatherApiErr.create(SMissingCityID);
  result := 'city_id=' + id;
end;

function StationIdLocation(const id: string): string;
begin
  if id = '' then
    Raise TWeatherApiErr.create(SMissingStationID);
  result := 'station=' + id;
end;


function WeatherUrl(q: TRequestElement;
                         const location: string;
                         const protocol: TGetProtocol;
                         const dataFormat: TDataFormat;
                         const units: TUnits;
                         const lang: string): string;

const
  requeststr: array[TRequestElement] of string = (
  {reCurrentCondition} 'current',
  {re120HourForecast}  'forecast/hourly',
  {re5DayForecast}     'forecast/3hourly',
  {re16DayForecast}    'forecast/daily',
  {reUsage}            'subscription/usage');

  protocolstr: array[TGetProtocol] of string = ('https', 'http');

  unitsstr: array[TUnits] of string = ('I', 'M', 'S');

begin
  result := Format(wburl, [protocolstr[protocol], requeststr[q]]);
  if q <> reUsage then begin
    if location = '' then
      Raise TWeatherApiErr.create(SMissingLocation);
    result := result + AnsiReplaceStr(location, ' ', '+');
    if lang <> '' then
      result := result + '&lang=' + lang;
    if units <> uMetric then
      result := result + '&units=' + unitsstr[units];
    result := result + '&';
  end;
  result := result + 'key={key}';
end;

end.


