unit owm;

{  Retrieve OpenWeatherMap (https://openweathermap.org) current
   weather conditions, weather forecasts and usage statistics
   as raw text (JSON formatted) and as TJSONData object. }

{ OpenWeatherMap API references for free accounts:

    Current weather API
       https://openweathermap.org/current

    5 day forecast API
       https://openweathermap.org/forecast5

    Subscription limits API
       https://openweathermap.org/appid

}


{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, strutils, dateutils,
    fpjson, laz2_dom;

type
    { Application exceptions }
  TWeatherApiErr = class(Exception);

    { Weather elements that  can be requested from OpenWeatherMap
      with free account. }
  TRequestElement = (reCurrentCondition,  // the current weather conditions
                     re5DayForecast,      // 3 hourly forecast for next 5 days
                     re7DayForecast);     // daily forecast for next 7 days

  TUnits = (uImperial, uMetric, uSI);

  TDataFormat = (dfJSON, dfXML);
  TGetProtocol = (gpHTTPS, gpHTTP);

function CityByNameLocation(const city: string; const country: string = ''): string;
function LatLongLocation(latitude, longitude: double): string;
function PostalCodeLocation(const code: string; const country: string = ''): string;
function CityIdLocation(const id: string): string;

function WeatherUrl(q: TRequestElement;
                         const location: string;
                         const protocol: TGetProtocol = gpHTTPS;
                         const dataFormat: TDataFormat = dfJSON;
                         const units: TUnits = uMetric;
                         const lang: string = ''): string;


implementation

uses
  owm_consts;

const
  owmurl = '%s://api.openweathermap.org/data/2.5/%s?';

function CityByNameLocation(const city, country: string): string;
begin
  if city = '' then
    Raise TWeatherApiErr.create(SMissingCityName);
  result := 'q=' + city;
  if country <> '' then
    result := result + ',' + country;
end;

function LatLongLocation(latitude, longitude: double): string;
var
  defaultDecimalSeparator: char;
begin
  defaultDecimalSeparator := DefaultFormatSettings.DecimalSeparator;
  DefaultFormatSettings.DecimalSeparator := '.';
  result := Format('lat=%.3f&lon=%.3f', [latitude, longitude]);
  DefaultFormatSettings.DecimalSeparator := defaultDecimalSeparator;
end;

function PostalCodeLocation(const code, country: string): string;
begin
  if code = '' then
    Raise TWeatherApiErr.create(SMissingPostalCode);
  result := 'zip=' + code;
  if country <> '' then
    result := result + ','+country;
end;

function CityIdLocation(const id: string): string;
begin
  if id = '' then
    Raise TWeatherApiErr.create(SMissingCityID);
  result := 'id=' + id;
end;

function WeatherUrl(q: TRequestElement;
                         const location: string;
                         const protocol: TGetProtocol;
                         const dataFormat: TDataFormat;
                         const units: TUnits;
                         const lang: string): string;

const
  requeststr: array[TRequestElement] of string = (
  {reCurrentCondition} 'weather',
  {re5DayForecast}     'forecast',
  {re7DayForecast}     'forecast/daily');

  protocolstr: array[TGetProtocol] of string = ('https', 'http');

  unitsstr: array[TUnits] of string = ('imperial', 'metric', '');
                                             // default = Kelvin
begin
  result := Format(owmurl, [protocolstr[protocol], requeststr[q]]);
  if location = '' then
    Raise TWeatherApiErr.create(SMissingLocation);
  result := result + AnsiReplaceStr(location, ' ', '+');
  if lang <> '' then
    result := result + '&lang=' + lang;
  if units <> uSI then
      result := result + '&units=' + unitsstr[units];
  if dataFormat =  dfXML then
      result := result + '&mode=xml'; // also supports html
  result := result + '&APPID={key}';
end;

end.


