unit darksky;

{  Retrieve DarkSky (https://DarkSky.org) current
   weather conditions, weather forecasts and usage statistics
   as raw text (JSON formatted) and as TJSONData object. }

{ DarkSky API reference :

    Dark Sky API — Overview
       https://darksky.net/dev/docs


    Free use is limited to 1000 request/day (reset at midnight UTC).

    My Domoticz server uses DarkSky and makes a request every 5 minutes
    which equals to a minimum of 288 calls/day. But looking at the
    forecast in Domoticz means additional calls are made.


}

{$mode objfpc}{$H+}

interface


uses
  Classes, SysUtils, strutils, dateutils, fpjson, laz2_dom;

type
    { Application exceptions }
  TWeatherApiErr = class(Exception);

    { Weather elements that  can be requested from DarkSky
      with free account. }
  TRequestElement = (reCurrentCondition,  // the current weather conditions
                     re48HourForecast,    // hourly forecast next 48 hours
                     re7DayForecast);     // daily forecast next 7 days

  TUnits = (uImperial, uMetric, uCanadaSI, uUkSI);

  TDataFormat = (dfJSON, dfXML);
  TGetProtocol = (gpHTTPS, gpHTTP);

function LatLongLocation(latitude, longitude: double): string;

function WeatherUrl(q: TRequestElement;
                         const location: string;
                         const protocol: TGetProtocol = gpHTTPS;
                         const dataFormat: TDataFormat = dfJSON;
                         const units: TUnits = uMetric;
                         const lang: string = ''): string;

implementation

uses
  darksky_consts, httprequests;

const
  apiUrl = '%s://api.darksky.net/forecast/{key}/{lat},{long}?%s';


function LatLongLocation(latitude, longitude: double): string;
var
  defaultDecimalSeparator: char;
begin
  defaultDecimalSeparator := DefaultFormatSettings.DecimalSeparator;
  DefaultFormatSettings.DecimalSeparator := '.';
  result := Format('%.3f,%.3f', [latitude, longitude]);
  DefaultFormatSettings.DecimalSeparator := defaultDecimalSeparator;
end;

function WeatherUrl(q: TRequestElement;
                         const location: string;
                         const protocol: TGetProtocol;
                         const dataFormat: TDataFormat;
                         const units: TUnits;
                         const lang: string): string;

const
  RequestExcludes: array[TRequestElement] of string = (
    'exclude=minutely,hourly,daily,alerts,flags',
    'exclude=currently,minutely,daily,alerts,flags',
    'exclude=currently,minutely,hourly,alerts,flags');

  protocolstr: array[TGetProtocol] of string = ('https', 'http');

  UnitsStr: array[TUnits] of string = ('us', 'si', 'ca', 'uk2');

begin
  result := Format(apiUrl, [Protocolstr[protocol], RequestExcludes[q]]);
  if location = '' then
    Raise TWeatherApiErr.create(SMissingLocation);
  result := AnsiReplaceStr(result, '{lat},{long}', location);
  if lang <> '' then
    result := result + '&lang=' + lang;
  if units <> uImperial then
      result := result + '&units=' + UnitsStr[units];
end;


end.


