(* This is based on LocalizedForms.pas from the example application


  This form can be the ancestor of other forms. It is suitable for changing
  languages at run-time. It provides the virtual method "UpdateTranslation"
  which must be overridden by each inherited form in order to update the
  translation of strings that are not automatically translated by
  DefaultTranslator (e.g. strings built up by means of Format statements).

  Please make sur clocale is added to the uses clause in the project
  file *.lpr or *.dpr:

      uses
      {$IFnDEF FPC}
      {$ELSE}
        clocale,
        Interfaces,
      {$ENDIF}
        Forms,
*)

unit LocalizedForms;

{$mode objfpc}{$H+}

interface

uses
  SysUtils, Forms;

type
  TLocalizedForm = class(TForm)
  protected
    procedure UpdateTranslation(const ALang: String = ''); virtual;
  public
  end;

var
  { This value is updated in SelectLanguage and in the initialization code.}
  CurrentLanguage: string = '';

procedure SelectLanguage(const ALang: string);

var
  enFormatSettings : TFormatSettings = (
    CurrencyFormat: 1;
    NegCurrFormat: 5;
    ThousandSeparator: ',';
    DecimalSeparator: '.';
    CurrencyDecimals: 2;
    DateSeparator: '-';
    TimeSeparator: ':';
    ListSeparator: ',';
    CurrencyString: '$';
    ShortDateFormat: 'd/m/y';
    LongDateFormat: 'dd" "mmmm" "yyyy';
    TimeAMString: 'AM';
    TimePMString: 'PM';
    ShortTimeFormat: 'hh:nn';
    LongTimeFormat: 'hh:nn:ss';
    ShortMonthNames: ('Jan','Feb','Mar','Apr','May','Jun',
                      'Jul','Aug','Sep','Oct','Nov','Dec');
    LongMonthNames: ('January','February','March','April','May','June',
                     'July','August','September','October','November','December');
    ShortDayNames: ('Sun','Mon','Tue','Wed','Thu','Fri','Sat');
    LongDayNames:  ('Sunday','Monday','Tuesday','Wednesday','Thursday','Friday','Saturday');
    TwoDigitYearCenturyWindow: 50;
  );

implementation

uses
  Classes, DefaultTranslator, LCLTranslator
  {$IFDEF Unix}
  , Unix, FileUtil
    {$IFDEF LCLCarbon}
  , MacOSAll
    {$ENDIF}
  {$ENDIF}
  {$IFDEF MSWINDOWS}
  , gettext
  {$ENDIF}
  ;

procedure UpdateBiDiMode(ALang: String);
begin
  if Application.IsRTLLang(ALang) then
    Application.BidiMode := bdRightToLeft
  else
    Application.BiDiMode := bdLeftToRight;
end;

procedure SelectLanguage(const ALang: string);
var
  i: Integer;
begin
  if ALang <> CurrentLanguage then begin
    SetDefaultLang(ALang);
    UpdateBiDiMode(ALang);
    CurrentLanguage := ALang;
    //SysUtils.GetFormatSettings; windows only!
    // so need to change DefaultFormatSettings in Linux;
    for i := 0 to Screen.FormCount-1 do
      if Screen.Forms[i] is TLocalizedForm then
        TLocalizedForm(Screen.Forms[i]).UpdateTranslation(ALang);
  end;
end;

{ Each inherited form will have to put code in procedure
  UpdateTranslation(ALang: String) to translate strings not
  automatically handled by DefaultTranslator.
  Don't forget to call "inherited". }
procedure TLocalizedForm.UpdateTranslation(const ALang: String);
begin
 if Application.IsRTLLang(ALang) <> Application.IsRTLLang(CurrentLanguage) then
   FlipChildren(true);
end;


function GetOSLanguage: string;
{platform-independent method to read the language of the user interface}
var
  l, fbl: string;
  {$IFDEF LCLCarbon}
  theLocaleRef: CFLocaleRef;
  locale: CFStringRef;
  buffer: StringPtr;
  bufferSize: CFIndex;
  encoding: CFStringEncoding;
  success: boolean;
  {$ENDIF}
begin
  {$IFDEF LCLCarbon}
  theLocaleRef := CFLocaleCopyCurrent;
  locale := CFLocaleGetIdentifier(theLocaleRef);
  encoding := 0;
  bufferSize := 256;
  buffer := new(StringPtr);
  success := CFStringGetPascalString(locale, buffer, bufferSize, encoding);
  if success then
    l := string(buffer^)
  else
    l := '';
  fbl := Copy(l, 1, 2);
  dispose(buffer);
  {$ELSE}
    {$IFDEF LINUX}
    fbl := Copy(GetEnvironmentVariable('LC_CTYPE'), 1, 2);
     if fbl = '' then
      fbl := Copy(GetEnvironmentVariable('LANG'), 1, 2);
    if fbl = '' then
      fbl := Copy(GetEnvironmentVariable('LANGUAGE'), 1, 2);
    {$ELSE} {Windows}
    l := '';
    fbl := '';
    GetLanguageIDs(l, fbl);
    {$ENDIF}
  {$ENDIF}
  Result := fbl;
end;


{ DefaultTranslator & LCLTranslator will load the locale & language specified
  by the user's environment. However, the user can change the language
  using the -l xx or --lang xx command line switches. In Windows and perhaps
  MAC OS X, command line switches are usually not case sensitive. So look at
  the switches again to pick up -L or --LANG }
//see  http://forum.lazarus.freepascal.org/index.php?topic=27326.0
function GetInitialLanguage: string;
{$IFnDEF Linux}
var
  i: integer;
{$ENDIF}
begin
  {$IFnDEF Linux}
  for i := 1 to ParamCount do
    if ( '--LANG' = upcase(paramstr(i)) ) or ('-L' = upcase(Paramstr(i)) ) then
    begin
       if i < ParamCount then begin
         result := lowercase(ParamStr(i+1));
         exit;
       end;
    end;
  {$ENDIF}
  result := CurrentLanguage;
end;

initialization
  CurrentLanguage := GetOSLanguage;
  SelectLanguage(GetInitialLanguage);
end.

