unit getweather;

{$mode objfpc}{$H+}


{ The weather information is not updated that frequently, so it makes
   sense to cache the information. That way looking at the raw data,
   the data as json formated information and then as text in
   quick succession will result in a single call to the server.

   Set the number of minutes that the cached information is kept
   with the CACHE_MINUTES constant defined below.}

{$DEFINE CACHE_RESPONSE}

interface

uses
  Classes, SysUtils, fpjson, laz2_dom;

{$IFDEF CACHE_RESPONSE}
const
  CACHE_MINUTES = 30;   // keep rawdata for 30 minutes, data seems to be updated every hour
{$ENDIF}


  { Returns true and the raw data obtained from weather server the
    variable rawdata if no error occurs. If an error occured, returns
    false and the error message is in the string variable rawdata.}
function WeatherRawData(const url: string; var rawdata: string): boolean;

function WeatherJSONData(const url: string; var rawdata: string; var data: TJSONData): boolean;
function WeatherXMLData(const url: string; var rawdata: string; var data: TXMLDocument): boolean;

resourcestring
  SYWMissingUrlErr = 'ERROR: No URL specified';
  SYWNoData = 'ERROR: Unable to retrieve data';
  SYWErrIncorrectJsonFormat = 'ERROR: Incorrect JSON format';
  SYWErrIncorrectXmlFormat = 'ERROR: Incorrect XML format';

implementation

uses
  {$IFDEF CACHE_RESPONSE} dateutils, {$ENDIF}
  httprequests, jsonparser, laz2_xmlread;

{$IFDEF CACHE_RESPONSE}
var
  cachedurl: string = '';
  cachedraw: string = '';
  cachetime: TDateTime = 0;
{$ENDIF}

function WeatherRawData(const url: string; var rawdata: string): boolean;
var
  code: integer;
begin
  if url = '' then begin
    rawdata := SYWMissingUrlErr;
    result := false;
    exit;
  end;

  {$IFDEF CACHE_RESPONSE}
  if now > cachetime then begin
    cachedurl := '';
    cachedraw := '';
  end
  else if url = cachedurl then begin
    rawdata := cachedraw;
    result := true;
    exit;
  end;
  {$ENDIF}

  rawdata := '';
  result := HTTPRequest(url, code, rawdata);
  if result and (rawdata = '') then begin
    rawdata := SYWNoData;
    result := false;
  end;

  {$IFDEF CACHE_RESPONSE}
  if result then begin
    cachedurl := url;
    cachedraw := rawdata;
    cachetime := IncMinute(now, CACHE_MINUTES);
  end;
  {$ENDIF}
end;


function WeatherJSONData(const url: string; var rawdata: string; var data: TJSONData): boolean;
begin
  data := nil;
  result := WeatherRawData(url, rawdata);
  if result then try
    data := GetJSON(rawdata);
    if not assigned(data) then
      Raise Exception.create(SYWErrIncorrectJsonFormat);
  except
    freeandnil(data);
    rawdata := SYWErrIncorrectJsonFormat;
    result := false;
  end;
end;

function WeatherXMLData(const url: string; var rawdata: string; var data: TXMLDocument): boolean;
var
  ss: TStringStream;
begin
  result := WeatherRawData(url, rawdata);
  if result then try
    ss := TStringStream.create(rawdata);
    try
      ss.Position := 0;
      readXmlFile(data, ss);
      if not assigned(data) then
        Raise Exception.create(SYWErrIncorrectXmlFormat);
    except
      freeandnil(data);
      rawdata := SYWErrIncorrectXmlFormat;
      result := false;
    end;
  finally
      ss.free;
  end;
end;


end.

