unit apixu;

{  Retrieve weather conditions and forecasts from
   Apixu (https://https://www.apixu.com/)
   as raw text (JSON formatted) and as TJSONData object. }

{ Apixu API reference :

     Documentation — Introduction
       https://www.apixu.com/doc/

     Attribution
       https://www.apixu.com/doc/credit.aspx

    Free use is limited to 

   
}

{$mode objfpc}{$H+}

interface


uses
  Classes, SysUtils, strutils, dateutils, fpjson, laz2_dom;

type
    { Application exceptions }
  TWeatherApiErr = class(Exception);

    { Weather elements that  can be requested from Apixu. }
   TRequestElement = (reCurrentCondition,  // the current weather conditions
                      reForecast);         // 10 day forecast

    { For display purposes, Apixu returns both }
  TUnits = (uImperial, uMetric);

  TDataFormat = (dfJSON, dfXML);
  TGetProtocol = (gpHTTPS, gpHTTP);

    { For AirportId location }
  TAirportCode = (acIATA, acICAO);

function CityByNameLocation(const city: string; const country: string = ''): string;
function LatLongLocation(latitude, longitude: double): string;
function PostalCodeLocation(const code: string): string;
function IpLocation(useAuto: boolean; const ip: string = ''): string;
function AirportIdLocation(const id: string; code: TAirportCode): string;

function WeatherUrl(q: TRequestElement;
                         const location: string;
                         const protocol: TGetProtocol = gpHTTPS;
                         const dataFormat: TDataFormat = dfJSON;
                         const units: TUnits = uMetric;
                         const lang: string = ''): string;



implementation

uses
  apixu_consts, httprequests,  localizedforms,
  fphttpclient {for encodeUrlElement};

const
  apiUrl = '%s://api.apixu.com/v1/{request}.{fmt}?key={key}&q={location}';


function CityByNameLocation(const city, country: string): string;
begin
  if city = '' then
    Raise TWeatherApiErr.create(SMissingCityName);
  result := 'city=' + city;
  if country <> '' then
    result := result + ',' + country;
end;

function LatLongLocation(latitude, longitude: double): string;
var
  defaultDecimalSeparator: char;
begin
  defaultDecimalSeparator := DefaultFormatSettings.DecimalSeparator;
  DefaultFormatSettings.DecimalSeparator := '.';
  result := Format('%.3f,%.3f', [latitude, longitude]);
  DefaultFormatSettings.DecimalSeparator := defaultDecimalSeparator;
end;

function PostalCodeLocation(const code: string): string;
begin
  if code = '' then
    Raise TWeatherApiErr.create(SMissingPostalCode);
  result := code;
  //if country <> '' then
    //result := result + ','+country;
end;

function IpLocation(useAuto: boolean; const ip: string): string;
begin
  if useAuto then
    result := 'auto:ip'
  else begin
    if ip = '' then
      Raise TWeatherApiErr.create(SMissingIpAddress);
    result := ip;
  end;
end;

function AirportIdLocation(const id: string; code: TAirportCode): string;
begin
  if id = '' then
    Raise TWeatherApiErr.create(SMissingAirportID);
  if code = acIATA then begin
    if length(id) <> 3 then
      Raise TWeatherApiErr.Create(SInvalidIataId);
    result := 'iata:' + id;
  end
  else begin
      if length(id) <> 4 then
        Raise TWeatherApiErr.Create(SInvalidIcaoId);
    result := 'metar:' + id;
  end;
end;

function WeatherUrl(q: TRequestElement;
                         const location: string;
                         const protocol: TGetProtocol;
                         const dataFormat: TDataFormat;
                         const units: TUnits;
                         const lang: string): string;

const
  RequestStr: array[TRequestElement] of string = (
    'current', 'forecast');

  formatStr: array[TDataFormat] of string = ('json', 'xml');

  protocolstr: array[TGetProtocol] of string = ('https', 'http');

begin
  if location = '' then
    Raise TWeatherApiErr.create(SMissingLocation);
  result := Format(apiUrl, [protocolstr[protocol]]);
  result := AnsiReplaceStr(result, '{fmt}', formatStr[dataFormat]);
  result := AnsiReplaceStr(result, '{request}', RequestStr[q]);
  result := AnsiReplaceStr(result, '{location}', EncodeURLElement(location));
  if q = reForecast then
    result := result + '&days=10';
  if lang <> '' then
    result := result + '&lang=' + lang;
  // units ignored!
end;


end.


