unit playunit;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

   { Starts a thread to play the specified file with the aplay utility.
     If the event handler is specified, it will be invoked when the
     playing stops. By default the process (ie aplay) and the owning thread are
     terminated when the application stops. Setting TerminateProcess to false
     will let aplay continue until the end of the file is reached.}
procedure Play(filename: string; PlayDone: TNotifyEvent = nil; TerminateProcess: boolean = true);

   {Will terminate the play process. If it was not active, this will have
    no effect.}
procedure StopPlaying;

implementation

uses
  process;

var
  StopPlay: boolean = false;
  AppClosing: boolean = false;

type
  TPlayThread = class(TThread)
  private
    FOnPlayDone: TNotifyEvent;
    FFilename: string;
    FTerminateProcess: boolean;
    procedure Done;
  protected
    procedure Execute; override;
  public
    constructor Create(const filename: string; PlayDone: TNotifyEvent = nil;
       TerminateProcess: boolean = true);
  end;

{ TPlayThread }

procedure TPlayThread.Done;
begin
  if assigned(FOnPlayDone) then
    FOnPlayDone(self);
end;

procedure TPlayThread.Execute;
var
  PlayProc: TProcess;
begin
  PlayProc := TProcess.create(nil);
  try
    PlayProc.executable := 'aplay';
    PlayProc.parameters.add(FFilename);
    PlayProc.Options := [poNoConsole];
    PlayProc.execute;
    while PlayProc.Running do begin
      if StopPlay or AppClosing then begin
        if StopPlay or FTerminateProcess then
          PlayProc.terminate(1);
        exit;
      end
      else
        sleep(1);
    end;
  finally
    PlayProc.free;
    if assigned(FOnPlayDone) and not AppClosing then
      synchronize(@Done);
  end;
end;

constructor TPlayThread.Create(const filename: string;  PlayDone: TNotifyEvent;
   TerminateProcess: boolean);
begin
  FOnPlayDone := PlayDone;
  FTerminateProcess := TerminateProcess;
  FFilename := filename;
  inherited create(false);
  FreeOnTerminate := true;
end;

procedure Play(filename: string; PlayDone: TNotifyEvent; TerminateProcess: boolean);
begin
  StopPlay := false;
  TPlayThread.create(filename, PlayDone, TerminateProcess);
end;

procedure StopPlaying;
begin
  StopPlay := true;
end;

finalization
  AppClosing := true;
end.

