unit TestConvFamily;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, fpcunit, testregistry, mdconvutils;

type

  { TTestConvFamilyCase }

  TTestConvFamilyCase = class(TTestCase)
  private
    FTestFamilies: array of string;
    FTestFamCount: integer;
    FFamilies: TConvFamilyArray;
    FDuplicateFam: integer;
    procedure RegisterTestFamilies;
  protected
    procedure SetUp; override;
    procedure TearDown; override;
    procedure RegisterEmptyString;
    procedure RegisterDuplicateDescription;
  published
    procedure TestHookUp;
    procedure TestRegisterConversionFamily;
    procedure TestUnregisterAllConversionFamilies;
    procedure TestConvFamilyToDescription;
    procedure TestDescriptionToConvFamily;
    procedure TestUnregisterFamily;
    procedure TestRegisterAfterUnregister;

    procedure TestEmptyDescriptionError;
    procedure TestRegisterDuplicateDescriptionError;
    procedure TestUnregisterUnknownFamilyError;
    procedure TestConvFamilyToDescriptionUnknownFamilyError;
  end;

  { TTestConvTypeCase }

  TTestConvTypeCase = class(TTestCase)
  private
    FConversionsRegistered: boolean;

    FTestFamilies: array of string;
    FTestFamCount: integer;
    FFamilies: TConvFamilyArray;

    FTestLengthDescriptions: array of string;
    FTestLengthConversions: array of TConvTypeInfo;
    FTestLengthCount: integer;
    FLengthConvTypes: TConvTypeArray;

    FTestTempDescriptions: array of string;
    FTestTempConversions: array of TConvTypeInfo;
    FTestTempCount: integer;
    FTemperatureConvTypes: TConvTypeArray;

    procedure RegisterTestConversions;
  protected
    procedure SetUp; override;
    procedure TearDown; override;
    procedure RegisterEmptyString;
    procedure RegisterDuplicateDescription;
    procedure UnregisterUnknown;
    procedure ConvTypeToDescriptionUnknownType;
    procedure RegisterWithUnknownFamily;
  published
    procedure TestHookUp;
    procedure TestRegisterConversionTypes;
    procedure TestConvTypeToDescription;
    procedure TestDescriptionToConvType;
    procedure TestLengthIdentityConversions;
    procedure TestTemperatureIdentityConversions;
    procedure TestLengthConversions;
    procedure TestTemperatureConversions;
    procedure TestUnregisterConversion;
    procedure TestRegisterAfterUnregister;

    procedure TestEmptyDescriptionWithNoFamiliesError;
    procedure TestEmptyDescriptionError;
    procedure TestRegisterDuplicateDescriptionError;
    procedure TestUnregisterUnknownConversionError;
    procedure TestConvTypeToDescriptionUnknownConversionError;
    procedure TestRegisterWithUnknownFamilyError;
  end;

implementation

{ TTestConvFamilyCase }

procedure TTestConvFamilyCase.RegisterTestFamilies;
var
  i: integer;
begin
  for i := 0 to FTestFamCount-1 do
    AssertEquals(i, RegisterConversionFamily(FTestFamilies[i]));
end;

procedure TTestConvFamilyCase.SetUp;
begin
  inherited;
  FTestFamCount := 3;
  setlength(FTestFamilies, FTestFamCount);
  FTestFamilies[0] := 'Length';
  FTestFamilies[1] := 'Weight';
  FTestFamilies[2] := 'Area';
  // make sure there are no registered conversion families and types;
  UnregisterAllConversionFamilies;
  setlength(FFamilies, 0);
end;

procedure TTestConvFamilyCase.TearDown;
begin
  FTestFamCount := 0;
  setlength(FTestFamilies, FTestFamCount);
  UnregisterAllConversionFamilies;
  setlength(FFamilies, 0);
  inherited;
end;

procedure TTestConvFamilyCase.RegisterEmptyString;
begin
  RegisterConversionFamily('  ');
end;

procedure TTestConvFamilyCase.RegisterDuplicateDescription;
begin
  RegisterConversionFamily(FTestFamilies[FDuplicateFam]);
end;


procedure TTestConvFamilyCase.TestHookUp;
begin
  AssertEquals('Start of test', 0, length(FFamilies));
  GetConvFamilies(FFamilies);
  AssertEquals('Should be empty at first', 0, length(FFamilies));
end;

procedure TTestConvFamilyCase.TestRegisterConversionFamily;
begin
  RegisterTestFamilies;
  GetConvFamilies(FFamilies);
  AssertEquals(FTestFamCount, length(FFamilies));
end;

procedure TTestConvFamilyCase.TestUnregisterAllConversionFamilies;
begin
  RegisterTestFamilies;
  GetConvFamilies(FFamilies);
  AssertEquals(FTestFamCount, length(FFamilies));

  UnregisterAllConversionFamilies;
  GetConvFamilies(FFamilies);
  AssertEquals(0, length(FFamilies));
end;

procedure TTestConvFamilyCase.TestConvFamilyToDescription;
var
  i: integer;
begin
  RegisterTestFamilies;
  for i := 0 to FTestFamCount-1 do
    AssertEquals(FTestFamilies[i], ConvFamilyToDescription(i));
end;

procedure TTestConvFamilyCase.TestDescriptionToConvFamily;
var
  i: integer;
  aFam: TConvFamily;
begin
  RegisterTestFamilies;
  for i := FTestFamCount-1 downto 0 do begin
    AssertTrue(FTestFamilies[i], DescriptionToConvFamily(FTestFamilies[i], aFam));
    AssertEquals('TConvFamily', i, aFam);
  end;
end;

procedure TTestConvFamilyCase.TestUnregisterFamily;
begin
  RegisterTestFamilies;

  UnregisterConversionFamily(1); // unregister weight
  GetConvFamilies(FFamilies);
  AssertEquals(FTestFamCount-1, length(FFamilies));
  AssertEquals(FTestFamilies[0], 0, FFamilies[0]);
  AssertEquals(FTestFamilies[2], 2, FFamilies[1]);

  UnregisterConversionFamily(0); // unregister length
  GetConvFamilies(FFamilies);
  AssertEquals(1, length(FFamilies));
  AssertEquals(FTestFamilies[2], 2, FFamilies[0]);

  UnregisterConversionFamily(2); // unregister area
  GetConvFamilies(FFamilies);
  AssertEquals(0, length(FFamilies));
end;

procedure TTestConvFamilyCase.TestRegisterAfterUnregister;
const
  SVolume = 'Volume';
begin
  RegisterTestFamilies;

  UnregisterConversionFamily(1); // unregister weight
  GetConvFamilies(FFamilies);
  AssertEquals(FTestFamCount-1, length(FFamilies));
  AssertEquals(FTestFamilies[0], 0, FFamilies[0]);
  AssertEquals(FTestFamilies[2], 2, FFamilies[1]);

  AssertEquals(1, RegisterConversionFamily(SVolume));

  GetConvFamilies(FFamilies);
  AssertEquals(FTestFamCount, length(FFamilies));
  AssertEquals(FTestFamilies[0], 0, FFamilies[0]);
  AssertEquals(SVolume, 1, FFamilies[1]);
  AssertEquals(FTestFamilies[2], 2, FFamilies[2]);

end;

procedure TTestConvFamilyCase.TestEmptyDescriptionError;
begin
  AssertException(EConversionError, @RegisterEmptyString, SInvalidFamilyDescription);
end;

procedure TTestConvFamilyCase.TestRegisterDuplicateDescriptionError;
begin
  RegisterTestFamilies;
  FDuplicateFam := 1;
  AssertException(EConversionError, @RegisterDuplicateDescription,
    Format(SDuplicateFamily, [FTestFamilies[FDuplicateFam]]) );
end;

procedure TTestConvFamilyCase.TestUnregisterUnknownFamilyError;
var
  i: integer;
begin
  RegisterTestFamilies;
  UnregisterConversionFamily(FTestFamCount+5);
  // check that nothing happened!
  GetConvFamilies(FFamilies);
  AssertEquals(FTestFamCount, length(FFamilies));
  for i := 0 to FTestFamCount-1 do
    AssertEquals(i, FFamilies[i]);
end;

procedure TTestConvFamilyCase.TestConvFamilyToDescriptionUnknownFamilyError;
begin
  RegisterTestFamilies;
  AssertEquals('', ConvFamilyToDescription(FTestFamCount + 8));
end;

{ TTestConvTypeCase }

function FahrenheitToCelsius(const AValue: Double): Double;
begin
  result:= (5.0/9.0) * (Avalue - 32.0);
end;

function CelsiusToFahrenheit(const AValue: Double): Double;
begin
  result:= 32.0 + ((9.0 * AValue)/ 5.0);
end;

procedure TTestConvTypeCase.SetUp;
begin
  inherited SetUp;
  FTestFamCount := 2;
  setlength(FTestFamilies, FTestFamCount);
  FTestFamilies[0] := 'Length';
  FTestFamilies[1] := 'Temperature';
  setlength(FFamilies, 0);

  FTestLengthCount := 3;
  setlength(FTestLengthDescriptions, FTestLengthCount);
  FTestLengthDescriptions[0] :=  'meter';
  FTestLengthDescriptions[1] :=  'inch';
  FTestLengthDescriptions[2] :=  'centimeter';
  setlength(FTestLengthConversions, FTestLengthCount);
  FTestLengthConversions[0] :=  TConvTypeFactor.Create(0, FTestLengthDescriptions[0], 1);
  FTestLengthConversions[1] :=  TConvTypeFactor.Create(0, FTestLengthDescriptions[1], 0.0254);
  FTestLengthConversions[2] :=  TConvTypeFactor.Create(0, FTestLengthDescriptions[2], 0.01);
  setlength(FLengthConvTypes, 0);

  FTestTempCount := 2;
  setlength(FTestTempDescriptions, FTestTempCount);
  FTestTempDescriptions[0] :=  'celsius';
  FTestTempDescriptions[1] :=  'fahrenheit';
  setlength(FTestTempConversions, FTestTempCount);
  FTestTempConversions[0] := TConvTypeFactor.Create(1, FTestTempDescriptions[0], 1);
  FTestTempConversions[1] := TConvTypeProcs.Create(1, FTestTempDescriptions[1],
                      TConversionProc(@CelsiusToFahrenheit),
                      TConversionProc(@FahrenheitToCelsius));
  setlength(FTemperatureConvTypes, 0);

  // make sure there are no registered conversion families and types;
  UnregisterAllConversionFamilies;
  // flag showing the Test conversions have not yet been registered;
  FConversionsRegistered := false;
end;

procedure TTestConvTypeCase.TearDown;
var
  i: integer;
begin
  UnregisterAllConversionFamilies;
  // in principle all conversion types have been freed unless they
  // were not registered in the first place!
  if not FConversionsRegistered then begin
    for i := 0 to FTestLengthCount-1 do
      FTestLengthConversions[i].free;
    for i := 0 to FTestTempCount-1 do
      FTestTempConversions[i].free;
  end;

  FTestFamCount := 0;
  FTestLengthCount := 0;
  FTestTempCount := 0;
  setlength(FTestFamilies, FTestFamCount);
  setlength(FLengthConvTypes, FTestLengthCount);
  setlength(FTestLengthDescriptions, FTestLengthCount);
  setlength(FTestLengthConversions, FTestLengthCount);
  setlength(FTemperatureConvTypes, FTestTempCount);
  setlength(FTestTempDescriptions, FTestTempCount);
  setlength(FTestTempConversions, FTestTempCount);
  FConversionsRegistered := false;
  inherited TearDown;
end;

procedure TTestConvTypeCase.RegisterEmptyString;
begin
  RegisterConversionType(0, '  ', 5);
end;

procedure TTestConvTypeCase.RegisterDuplicateDescription;
begin
  RegisterConversionType(0, FTestLengthDescriptions[0], 5);
end;

procedure TTestConvTypeCase.UnregisterUnknown;
begin
   UnregisterConversionType(2346);
end;

procedure TTestConvTypeCase.ConvTypeToDescriptionUnknownType;
begin
  ConvTypeToDescription(2346);
end;

procedure TTestConvTypeCase.RegisterWithUnknownFamily;
begin
  RegisterConversionType(32, 'lbs', 1/2.2);
end;

procedure TTestConvTypeCase.RegisterTestConversions;
var
  i: integer;
  aType: TConvType;
begin
  for i := 0 to FTestFamCount-1 do
    AssertEquals(i, RegisterConversionFamily(FTestFamilies[i]));
  for i := 0 to FTestLengthCount-1 do begin
    AssertTrue(RegisterConversionType(FTestLengthConversions[i], aType));
    AssertEquals(i, aType);
  end;
  for i := 0 to FTestTempCount-1 do begin
    AssertTrue(RegisterConversionType(FTestTempConversions[i], aType));
    AssertEquals(FTestLengthCount+i, aType);
  end;
  FConversionsRegistered := true;
end;

procedure TTestConvTypeCase.TestHookUp;
begin
  AssertEquals('Start of test', 0, length(FFamilies));
  GetConvFamilies(FFamilies);
  AssertEquals('No families at first', 0, length(FFamilies));

  GetConvTypes(0, FLengthConvTypes);
  AssertEquals('No length conversions at first', 0, length(FLengthConvTypes));

  GetConvTypes(1, FTemperatureConvTypes);
  AssertEquals('No temperature conversions at first', 0, length(FTemperatureConvTypes));
end;

procedure TTestConvTypeCase.TestRegisterConversionTypes;
begin
  RegisterTestConversions;

  GetConvFamilies(FFamilies);
  AssertEquals(FTestFamCount, length(FFamilies));

  GetConvTypes(0, FLengthConvTypes);
  AssertEquals(FTestLengthCount, 0, length(FLengthConvTypes));

  GetConvTypes(1, FTemperatureConvTypes);
  AssertEquals(FTestTempCount, length(FTemperatureConvTypes));
end;

procedure TTestConvTypeCase.TestConvTypeToDescription;
var
  i: integer;
begin
  RegisterTestConversions;
  GetConvTypes(0, FLengthConvTypes);
  for i := 0 to length(FLengthConvTypes)-1 do
    AssertEquals(FTestLengthDescriptions[i], ConvTypeToDescription(FLengthConvTypes[i]));

  GetConvTypes(1, FTemperatureConvTypes);
  for i := 0 to length(FTemperatureConvTypes)-1 do
    AssertEquals(FTestTempDescriptions[i], ConvTypeToDescription(FTemperatureConvTypes[i]));
end;

procedure TTestConvTypeCase.TestDescriptionToConvType;
var
  i: integer;
  aType: TConvType;
begin
  RegisterTestConversions;
  for i := 0 to FTestLengthCount-1 do begin
    AssertTrue(DescriptionToConvType(0, FTestLengthDescriptions[i], aType));
    AssertEquals(i, aType);
  end;
  for i := 0 to FTestTempCount-1 do begin
    AssertTrue(DescriptionToConvType(1, FTestTempDescriptions[i], aType));
    AssertEquals(FTestLengthCount+i, aType);
  end;
end;

procedure TTestConvTypeCase.TestLengthIdentityConversions;
var
  i: integer;
begin
  RegisterTestConversions;
  GetConvTypes(0, FLengthConvTypes);
  for i := 0 to length(FLengthConvTypes)-1 do
    AssertEquals(1.2, Convert(1.2, FLengthConvTypes[i], FLengthConvTypes[i]));
end;

procedure TTestConvTypeCase.TestTemperatureIdentityConversions;
var
  i: integer;
begin
  RegisterTestConversions;
  GetConvTypes(1, FTemperatureConvTypes);
  for i := 0 to length(FTemperatureConvTypes)-1 do
    AssertEquals(1.2, Convert(1.2, FTemperatureConvTypes[i], FTemperatureConvTypes[i]));
end;

procedure TTestConvTypeCase.TestLengthConversions;
var
  metertype, centimetertype, inchtype: TConvType;
begin
  RegisterTestConversions;
  AssertTrue(DescriptionToConvType(FTestLengthDescriptions[0], meterType));
  AssertTrue(DescriptionToConvType(FTestLengthDescriptions[1], inchType));
  AssertTrue(DescriptionToConvType(FTestLengthDescriptions[2], centimeterType));
  AssertEquals(0, meterType);
  AssertEquals(1, inchType);
  AssertEquals(2, centimeterType);

  AssertEquals(250, Convert(2.5, meterType, centimeterType));
  AssertEquals(0.1, Convert(10, centimeterType, meterType));
  AssertEquals(2.54, Convert(1, inchType, centimeterType));
end;

procedure TTestConvTypeCase.TestTemperatureConversions;
var
  degCtype, degFtype: TConvType;
begin
  RegisterTestConversions;
  AssertTrue(DescriptionToConvType(FTestTempDescriptions[0], degCType));
  AssertTrue(DescriptionToConvType(FTestTempDescriptions[1], degFType));
  AssertEquals(3, degCType);
  AssertEquals(4, degFType);
  AssertEquals(0, Convert(32, degFType, degCType));
  AssertEquals(100, Convert(212, degFType, degCType));
  AssertEquals(-15, Convert(5, degFType, degCType));
  AssertEquals(32, Convert(0, degCType, degFType));
  AssertEquals(212, Convert(100, degCType, degFType));
  AssertEquals(41, Convert(5, degCType, degFType));
end;

procedure TTestConvTypeCase.TestUnregisterConversion;
var
  i: integer;
  aType: TConvType;
begin
  RegisterTestConversions;

  AssertTrue(DescriptionToConvType(FTestLengthDescriptions[2], aType));
  UnregisterConversionType(aType); // unregister centimeter

  GetConvTypes(0, FLengthConvTypes);
  AssertEquals(2, length(FLengthConvTypes));
  for i := 0 to 1 do begin
    AssertTrue(DescriptionToConvType(0, FTestLengthDescriptions[i], aType));
    AssertEquals(i, aType);
  end;
end;

procedure TTestConvTypeCase.TestRegisterAfterUnregister;
var
  i: integer;
  aType: TConvType;
begin
  RegisterTestConversions;

  AssertTrue(DescriptionToConvType(FTestLengthDescriptions[1], aType));
  UnregisterConversionType(aType); // unregister inch
  GetConvTypes(0, FLengthConvTypes);
  AssertEquals(2, length(FLengthConvTypes));

  // slot 1 now empty
  // should be re-used
  FTestLengthDescriptions[1] :=  'yard';
  FTestLengthConversions[1] :=  TConvTypeFactor.Create(0, FTestLengthDescriptions[1], 0.9144);

  AssertTrue(RegisterConversionType(FTestLengthConversions[1], aType));
  AssertEquals(1, aType);

  GetConvTypes(0, FLengthConvTypes);
  AssertEquals(FTestLengthCount, length(FLengthConvTypes));
  for i := 0 to FTestLengthCount-1 do begin
    AssertTrue(DescriptionToConvType(0, FTestLengthDescriptions[i], aType));
    AssertEquals(i, aType);
  end;
end;

procedure TTestConvTypeCase.TestEmptyDescriptionWithNoFamiliesError;
begin
  AssertException(EConversionError, @RegisterEmptyString,
    Format(SCannotRegisterConversion, ['  ', '']));
end;


procedure TTestConvTypeCase.TestEmptyDescriptionError;
begin
  RegisterTestConversions;

  AssertException(EConversionError, @RegisterEmptyString,
    Format(SCannotRegisterConversion, ['  ', FTestFamilies[0]]));
end;

procedure TTestConvTypeCase.TestRegisterDuplicateDescriptionError;
begin
  RegisterTestConversions;
  AssertException(EConversionError, @RegisterDuplicateDescription,
    Format(SCannotRegisterConversion, [FTestLengthDescriptions[0], FTestFamilies[0]]));
end;

procedure TTestConvTypeCase.TestUnregisterUnknownConversionError;
begin
  RegisterTestConversions;
  AssertException(EConversionError, @UnregisterUnknown,
    Format(SUnknownUnit, [2346]));
end;

procedure TTestConvTypeCase.TestConvTypeToDescriptionUnknownConversionError;
begin
  RegisterTestConversions;
  AssertException(EConversionError, @ConvTypeToDescriptionUnknownType,
    Format(SUnknownUnit, [2346]));
end;

procedure TTestConvTypeCase.TestRegisterWithUnknownFamilyError;
begin
  RegisterTestConversions;
  AssertException(EConversionError, @RegisterWithUnknownFamily,
    Format(SCannotRegisterConversion, ['lbs', '']));
end;


initialization
  RegisterTest(TTestConvFamilyCase);
  RegisterTest(TTestConvTypeCase);
end.

