{
   This file is part of the Free Pascal run time library.
   Copyright (c) 2004 by Marco van de Voort
   member of the Free Pascal development team.

   An implementation for unit convutils, which converts between
   units and simple combinations of them.

   Based on a guessed interface derived from some programs on the web. (Like
   Marco Cantu's EuroConv example), so things can be a bit Delphi
   incompatible. Also part on Delphibasics.co.uk.

   Quantities are mostly taken from my HP48g/gx or the unix units program

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY;without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**********************************************************************
  
  Improved implementation which makes it possible to include non
  linear conversions such as between the Fahrenheit and Celsius 
  temperature scales.
  
  The interface part of the original file had the correct structures
  TConvTypeInfo, TConvTypeFactor and especially TConvTypeProcs to handle 
  conversions that were more complicated than simple ratios, but they were
  not used in the implementation part. Accordingly the implementation 
  part is much changed.
  
  Also added some additional procedures and functions for better 
  Delphi compatibility.
  
  The accompanying stdconvs.pp is also modified to take advantage of
  these changes.
   
  Michel Deslierres, 
  http://www.sigmdel.ca/michel/program/fpl/conversion/conversions_en.html

  30 August 2016
}

unit mdConvUtils;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

type
    { Type of the unique identier assigned to conversion families (i.e measures)
      such as length, time, volume, etc. }
  TConvFamily = integer;

    { Type of the unique identifier assigned to conversion types (i.e. units
      of a measure) used in a measurement such as cm and feet that are distance
      measures,  hours and seconds that are time measures and so on). }
  TConvType = integer;

    { Dynamic array of conversion family identifiers. }
  TConvFamilyArray = array of TConvFamily;

    { Dynamic array of conversion types identifiers. }
  TConvTypeArray = array of TConvType;

    { Floating point type used for conversion results.}
  TConvUtilFloat = double;

    { Exceptions raised in this unit. }
  EConversionError = class(Exception);

type
  { TConvTypeInfo }

    { Abstract parent class of all conversions of units.

      All conversions between units used for measuring a quantity, say
      distance or weight, are done by first converting to a base unit
      (meter in distance, °Celsius in temperature and so on) and then from the
      base unit to the target unit.

      So the conversion from x inches to w millimeters, means that
      at first x inches are converted to y meters and then y meters are
      converted to w millimeters. Accordingly each conversion must know
      how to convert values measured in the base unit to its own unit
      and how to convert quantities measured in its own unit to the
      base unit.}
  TConvTypeInfo = class
  private
    FDescription: string;
    FConvFamily: TConvFamily;
    FConvType	  : TConvType;
  public
    constructor Create(const aFamily: TConvFamily; const aDescription: string);

      { Converts a quantity in the base unit to the matching quantity
        in the conversion's unit. }
    function FromCommon(aValue: TConvUtilFloat): TConvUtilFloat; virtual; abstract;

      { Converts a quantity in the conversion's unit to the matching
        quantity in the base unit. }
    function ToCommon(aValue: TConvUtilFloat): TConvUtilFloat; virtual; abstract;

      { Name of the conversion's unit.}
    property Description: string read FDescription;
   
      { The conversion type's identifier. }  
    property ConvType: TConvType read FConvType;

      { Physical quantity being measured by the unit. This field is used to
        verify that the units are compatible before attempting a
        conversion. See the function Convert.}
    property ConvFamily: TConvFamily read FConvFamily;
  end;

  { TConvTypeFactor }

    { Most conversions are linear functions, converting between units
      by multiplying or dividing by a constant called Factor. Typical
      usage:
        TFactorConversion.Create('Kilometers', 1000)
      The created conversion's FromBase and ToBase functions will then
      know how to convert kilometers to meters and meters to kilometers.
        ToBase:   y kilometers = 1000 * y meters
        FromBase: w meters = w / 1000 kilometers. }
  TConvTypeFactor = class(TConvTypeInfo)
  private
    FFactor: TConvUtilFloat;
  public
      { The name of the scale and the factor to convert measurements to
        the base scale are needed to create a conversion. }
    constructor Create(const aFamily: TConvFamily; const aDescription: string;
                       const aFactor: TConvUtilFloat);

      { Implementation of the conversion methods}
    function FromCommon(aValue: TConvUtilFloat): TConvUtilFloat; override;
    function ToCommon(aValue: TConvUtilFloat): TConvUtilFloat; override;
  end;

  { TConvTypeProcs }

    { Some conversions are more complicated, converting Fahrenheit degrees
     to Celsius is an affine function, not a linear function. }
  TConversionProc  = function(const AValue: TConvUtilFloat): TConvUtilFloat;

    { Creating a "complicated" conversion involves specifying two
      transformations (i.e. TConversionProc) to and from the base
      unit.  Typical usage:

        TFunctionConversion.Create('Fahrenheight',
          TConversionProc(@CelsiusToFahrenheit),
          TConversionProc(@FahrenheitToCelsius))

      The created conversion's FromCommon and ToCommon functions will then
      call on the two specifed converted functions to convert Fahrenheit degrees
      to Celsius degreed and back
        ToCommon:   y °F = FahrenheitToCelsius(y) °C
        FromCommon: w °C = CelsiusToFahrentheit(w) °F.
      }
  TConvTypeProcs = class(TConvTypeInfo)
  private
    FFromBase: TConversionProc;
    FToBase: TConversionProc;
  public
      { The name of the unit and the factor to convert measurements to
        the base unit are needed to create a conversion. }
    constructor create(const aFamily: TConvFamily; const aDescription: string;
                       const aFromBase, aToBase: TConversionProc);

      { Implementation of the conversion methods}
    function FromCommon(aValue: TConvUtilFloat): TConvUtilFloat; override;
    function ToCommon(aValue: TConvUtilFloat): TConvUtilFloat; override;
  end;

  { Registers a conversion family, which is a measurable quantity such as
    length or temperatur. The function returns a new family identifier if the
    description is not found among the descriptions of previously registered
    families, otherwise it returns the family identifier under which it was
    previously registered.

    If the description is an empty string, a CIllegalConvFamily identifier
    is returned.}
function RegisterConversionFamily(const aDescription: string): TConvFamily;

  { Removes a previously registered conversion family. Once a conversion
    family is removed, all conversions between units of that family
    become illegal. Note that registering a family after unregistering it
    may cause problems because the new family identifier may not be
    the original identifier.}
procedure UnregisterConversionFamily(aFamily: TConvFamily);

  { Essentially resets the unit. All memory allocated previously to
    conversions types and conversion families is reclaimed.}
procedure UnregisterAllConversionFamilies;

  { Returns a dynamic array of all currently registered conversion families.}
procedure GetConvFamilies(out AFamilies: TConvFamilyArray);

  { Returns the description from the conversion family identifier.}
function ConvFamilyToDescription(const AFamily: TConvFamily): string;

  { Returns true and the conversion family identifier in aFamily given a
    description if the latter matches an already registered conversion family.
    If there is no matching registered family returns false and aFamily is
    set to CIllegalConvFamily. Not case sensitive.}
function DescriptionToConvFamily(const aDescription: string;
                   out aFamily: TConvFamily): boolean;

{ Registers a conversion type, which is a measure or unit such as cm, foot,
  or liter. The function returns a new conversion type identifier if the
  description is not found among the descriptions of previously registered
  conversion type for the specified conversion family.

  Will raise an EConversionError if the description is an empty string or
  if the conversion family already contains  a conversion type with the same
  description.

  Use this version of the routine to create a simple conversion type
  (TConvTypeFactor) where conversion to and from the base unit is a linear
  function.}
function RegisterConversionType(const aFamily: TConvFamily;
                                const aDescription: String;
                                const aFactor: TConvUtilFloat): TConvType; overload;

{ Registers a conversion type, which is a measure or unit such as cm, foot,
  or liter. The function returns a new conversion type identifier if the
  description is not found among the descriptions of previously registered
  conversion type for the specified conversion family.

  Will raise an EConversionError if the description is an empty string or
  if the conversion family already contains  a conversion type with the same
  description.

  Use this version of the routine to create a complex conversion type
  (TConvTypeProcs) where conversion to and from the base unit is a not a
  linear function.}
function RegisterConversionType(const aFamily: TConvFamily;
                                const aDescription: string;
   															const aFromBase: TConversionProc;
                                const aToBase: TConversionProc): TConvType; overload;

{ Registers a conversion type, which is a measure or unit such as cm, foot,
  or liter. The function returns a new conversion type identifier if the
  description is not found among the descriptions of previously registered
  conversion type for the specified conversion family.

  Will raise an EConversionError if the aConversion.description is an empty
  string or if the conversion family (aConversion.ConvFamily) already contains
  a conversion type with the same description.

  This version of the routine is not usually directly used but is invoked
  by the other two versions.}
function RegisterConversionType(aConversion: TConvTypeInfo;
                                out aType: TConvType): boolean; overload;

{ Removes a conversion type from the list of available conversions. In other
  words, the GetConvTypes procedure will return an array of TConvType that
  will not contain aType. }
procedure UnregisterConversionType(const aType: TConvType);

{ Returns a dynamic array of all currently registered conversion types.}
procedure GetConvTypes(const aFamily: TConvFamily; out aTypes: TConvTypeArray);

{ Returns the description from the conversion type identifier.}
function ConvTypeToDescription(const aType: TConvType): string;

{ Returns the conversion type identifier of the first conversion type found
  with a description matching aDescription. Since the description "ounce" can
  be used for a unit of weight and a unit of volume, this is not the best
  version of the function.}
function DescriptionToConvType(const aDescription: string;
                               out aType: TConvType): Boolean; overload;

{ Returns the conversion type identifier of the first conversion type found
  with a description matching aDescription in the specified conversion
  family. It will therefore be possible to correctly obtain the convertion
  type for "ounce" even if the description is used with a unit of weight and
  with a unit of volume. If aFamily is set to CIllegalConvFamily, then
  this version of the function will act as the other version: the first
  conversion type found with a description matching aDescription will be
  returned.}
function DescriptionToConvType(const aFamily: TConvFamily;
                               const aDescription: string;
                               out aType: TConvType): Boolean; overload;

{ Returns true if the convert function will be able to perform a conversion
  between the two specified conversion types.}
function CompatibleConversionTypes(const aFrom, aTo: TConvType): Boolean;

{  Convert the value AValue from one set of measurement units to another.
   The FromUnit parameter specifies the current units of the measurement AValue,
   the ToUnit parameter specifies the units of measurement of the result.

   For example, Convert(2.8, duYards, duMeters) will return 2,56032 since
   2.8 yards is equal to 2.56032 meters.

   As the example shows, the aFrom and aTo conversion types must belong to
   the same conversion family; they must be measures of the same physical
   quantity. The function CompatibleConversionTypes can test that this is
   the case.

   These conversions types are defined in the StdConvs unit along with may
   others.}
function Convert(const aValue: TConvUtilFloat;
     const FromUnit, ToUnit: TConvType): TConvUtilFloat;

{  Convert the value AValue from one set of compound measurement units to another.
   The FromUnit1 and FromUnit2 parameters specify the current units of
   AValue. The ToUnit1 and ToUnit2 parameters specify the units of the result.

   As an example
     Speed := Convert(value, duMiles, tuHours, duMeters, tuSeconds);

   The units specified by FromUnit1 and ToUnit1 must belong to the same
   conversion family and the units specified by FromUnit2 and ToUnit2 must
   belong to same conversion family (which can of course be different from the
   first conversion family.}
function Convert(const AValue: TConvUtilFloat;
     const FromUnit1, FromUnit2,  ToUnit1, ToUnit2: TConvType): TConvUtilFloat;

{ Fills the given TStrings with the descriptions of all registered families.
  The corresponding object is actually each family's identifier.
  Typical usage: AssignFamilies(AComboBox.Items) to fill the
  combo box with the list of all conversions families.}
function AssignFamilies(Strings: TStrings): integer;

{ Fills the given TStrings with the descriptions of all registered conversion
  types for the given conversion family. The corresponding object of each 
  description is actually the convertion type's identifier. 
  
  Typical usage:  AssignConvTypes(CurrentFamily, AComboBox.Items) 
  which fills the combo box with the list of all conversions types for the 
  current conversion family.}
function AssignConvTypes(aFamily: TConvFamily; Strings: TStrings): integer;

const
  CIllegalConvFamily = -1;
  CIllegalConvType = -1;

ResourceString
  SCannotConvert = 'Cannot convert from %s to %s';
  SUnknownUnit = 'Unknown conversion unit (%d)';
  SDuplicateFamily = 'Conversion family %s already registered';
  SInvalidFamilyDescription = 'No conversion family description';
  SCannotRegisterConversion = 'Cannot register conversion "%s" in conversion family "%s"';

implementation

const
  // Number of slots added to Conversions and Families when full
  // Cuts down on reallocations.
  ConversionsDelta = 8;
  FamiliesDelta = 4;

var
  Conversions: array of TConvTypeInfo = nil;
  Families: array of string = nil;

procedure ClearConversions;
var
  i: integer;
begin
  for i := length(Conversions)-1 downto 0 do
    Conversions[i].Free;
  setlength(Conversions, 0);
end;

function RegisterConversionFamily(const aDescription: string): TConvFamily;
var
  i,l : Longint;
begin
  if trim(aDescription) = '' then
    Raise EConversionError.Create(SInvalidFamilyDescription);
  l := length(Families);
  i := 0;
  // check for duplicate
  while (i < l) and (aDescription <> Families[i]) do
    inc(i);
  if i < l then
    Raise EConversionError.CreateFmt(SDuplicateFamily, [aDescription]);
  // reuse any deleted slot
  i := 0;
  while (i < l) and (Families[i] <> '') do
    inc(i);
  if i = l then begin // no deleted slot found, add some to the array
    SetLength(Families, l + FamiliesDelta);
    while l < length(Families) do begin
      Families[l] := '';
      inc(l);
    end;
  end;
  families[i] := aDescription;
  Result := i;
end;

procedure UnregisterConversionFamily(aFamily: TConvFamily);
var
  i: integer;
begin
  // mark a registered family as unregistered by erasing its description
  if (0 <= aFamily) and (aFamily < length(Families)) then begin
    Families[aFamily] := '';
    for i := 0 to length(Conversions)-1 do
      if assigned(Conversions[i]) and (Conversions[i].ConvFamily = aFamily) then begin
        Conversions[i].free;
        Conversions[i] := nil;
      end;
  end;
end;

procedure UnregisterAllConversionFamilies;
begin
  setlength(Families, 0);
  ClearConversions;
end;

procedure GetConvFamilies(out AFamilies: TConvFamilyArray);
var
  l,i,j : integer;
begin
  l := length(Families);
  setlength(AFamilies, l);
  j := 0;
  for i := 0 to length(Families)-1 do begin
    if Families[i] = '' then
      dec(l)
    else begin
      AFamilies[j] := i;
      inc(j);
    end;
  end;
  setlength(aFamilies, l);
end;

function ConvFamilyToDescription(const AFamily: TConvFamily): string;
begin
  if (0 <= aFamily) and (aFamily < length(Families)) then
    result := Families[aFamily]
  else
    result := '';
end;

function DescriptionToConvFamily(const aDescription: string;
                  out aFamily: TConvFamily): boolean;
var
  i: integer;
begin
  result := false;
  aFamily := CIllegalConvFamily;
  if trim(aDescription) <> '' then
    for i := length(Families)-1 downto 0 do
       if AnsiSameText(aDescription, Families[i]) then begin
         aFamily := i;
         result := true;
         break;
       end;
end;

{ TConvTypeInfo }

constructor TConvTypeInfo.Create(const aFamily: TConvFamily; const aDescription: string);
begin
  inherited Create;
  FDescription := aDescription;
  FConvFamily := aFamily;
end;

{ TConvTypeFactor }

constructor TConvTypeFactor.create(const aFamily: TConvFamily;
                  const aDescription: string; const aFactor: TConvUtilFloat);
begin
  inherited Create(aFamily, aDescription);
  FFactor := aFactor;
end;

function TConvTypeFactor.FromCommon(aValue: TConvUtilFloat): TConvUtilFloat;
begin
  result := aValue/FFactor;
end;

function TConvTypeFactor.ToCommon(aValue: TConvUtilFloat): TConvUtilFloat;
begin
  result := aValue*FFactor;
end;

{ TConvTypeProcs }

constructor TConvTypeProcs.create(const aFamily: TConvFamily;
                   const aDescription: string;
                   const aFromBase, aToBase: TConversionProc);
begin
  inherited Create(aFamily, aDescription);
  FFromBase := aFromBase;
  FToBase := aToBase;
end;

function TConvTypeProcs.FromCommon(aValue: TConvUtilFloat): TConvUtilFloat;
begin
  result := FFromBase(aValue);
end;

function TConvTypeProcs.ToCommon(aValue: TConvUtilFloat): TConvUtilFloat;
begin
  result := FToBase(aValue);
end;

{ global conversion routines }

function DescriptionToConvType(const aFamily: TConvFamily;
                               const aDescription: string;
                               out aType: TConvType): Boolean;
var
  i: integer;
begin
  aType := CIllegalConvType;
  result := (CIllegalConvFamily <= aFamily)
     and (aFamily < length(Families) )
     and (Families[aFamily] <> '')
     and (trim(aDescription) <> '');
  if result then begin
    for i := 0 to length(Conversions)-1 do begin
      if not assigned(Conversions[i])then
        continue;
      if ( (aFamily = CIllegalConvFamily)
           or (aFamily = Conversions[i].ConvFamily) )
      and AnsiSameText(aDescription, Conversions[i].Description) then begin
        aType := i;
        exit;
      end;
    end;
    result := false;
  end;
end;

function DescriptionToConvType(const aDescription: string;
                               out aType: TConvType): Boolean;
begin
  result := DescriptionToConvType(CIllegalConvType, aDescription, aType);
end;

function RegisterConversionType(aConversion: TConvTypeInfo;
     out aType: TConvType): boolean;
var
  i,l : Longint;
begin
  // make sure that aConverions was created, that its description is not
  // empty and that the conversion family exists.
  Result := assigned(aConversion) and (trim(aConversion.Description) <> '')
            and (0 <= AConversion.ConvFamily) and (AConversion.ConvFamily < length(Families))
            and (Families[AConversion.ConvFamily] <> '');
  // all that ok, make sure its not a duplicate
  if Result then
    Result := not DescriptionToConvType(aConversion.ConvFamily,
                     aConversion.Description, aType);
  if result then begin
    l := length(Conversions);
    i := 0;
    // search for a deleted slot
    while (i < l) and assigned(Conversions[i]) do
      inc(i);
    if (i = l) then begin // no deleted slot found, add some to the array
      SetLength(Conversions, l + ConversionsDelta);
      while l < length(Conversions) do begin
        Conversions[l] := nil;
        inc(l);
      end;
    end;
    Conversions[i] := aConversion;
    aConversion.FConvType := i;
    aType := i;
  end;
end;

function RegisterConversionType(const aFamily: TConvFamily;
                                const aDescription: String;
                                const aFactor: TConvUtilFloat): TConvType;
var
  aConversion: TConvTypeInfo;
begin
  aConversion := TConvTypeFactor.Create(aFamily, aDescription, aFactor);
  if not RegisterConversionType(aConversion, result) then begin
    aConversion.free;
    Raise EConversionError.CreateFmt(SCannotRegisterConversion,
       [aDescription, ConvFamilyToDescription(aFamily)]);
  end;

end;

function RegisterConversionType(const aFamily: TConvFamily;
                                const aDescription: string;
   															const aFromBase: TConversionProc;
                                const aToBase: TConversionProc): TConvType; overload;
var
  aConversion: TConvTypeInfo;
begin
  aConversion := TConvTypeProcs.Create(aFamily, aDescription, aFromBase, aToBase);
  if not RegisterConversionType(aConversion, result) then begin
    aConversion.free;
    Raise EConversionError.CreateFmt(SCannotRegisterConversion,
       [aDescription, ConvFamilyToDescription(aFamily)]);
  end;
end;

procedure GetConvTypes(const aFamily: TConvFamily; out ATypes: TConvTypeArray);
var
  i, n: Integer;
begin
  n := 0;
  for i := 0 to Length(Conversions) - 1 do
    if Assigned(Conversions[i]) and (Conversions[i].ConvFamily = AFamily) then
       inc(n);
  SetLength(aTypes, n);
  n := 0;
  for i := 0 to Length(Conversions) - 1 do
    if Assigned(Conversions[i]) and (Conversions[i].ConvFamily = AFamily) then begin
      aTypes[n] := i;
      inc(n);
    end;
end;

function GetConversion(aType: TConvType): TConvTypeInfo;
begin
  if (0 <= aType) and (aType < length(Conversions))
  and assigned(Conversions[aType]) then
      result := Conversions[aType]
  else
    Raise EConversionError.CreateFmt(SUnknownUnit, [aType]);
end;

procedure UnregisterConversionType(const AType: TConvType);
begin
  GetConversion(aType).Free;
  Conversions[aType] := nil;
end;

function ConvTypeToDescription(const aType: TConvType): string;
begin
  result := GetConversion(aType).Description;
end;

function ConvTypeToConvFamily(const aType: TConvType): TConvFamily;
begin
  result := GetConversion(aType).ConvFamily
end;

function CompatibleConversionTypes(const aFrom, aTo: TConvType): Boolean;
begin
  try
    result := ConvTypeToConvFamily(aFrom) = ConvTypeToConvFamily(aTo)
  except
    result := false;
  end;
end;

function Convert(const aValue: TConvUtilFloat;
                 const FromUnit, ToUnit: TConvType): TConvUtilFloat;
var
  FromConversion: TConvTypeInfo;
  ToConversion: TConvTypeInfo;
begin
  FromConversion := GetConversion(FromUnit);
  ToConversion := GetConversion(ToUnit);
  if FromConversion.ConvFamily <> ToConversion.ConvFamily then
    raise EConversionError.CreateFmt(SCannotConvert,
      [ConvTypeToDescription(FromUnit),
       ConvTypeToDescription(ToUnit)]);
  result := FromConversion.ToCommon(aValue);
  result := ToConversion.FromCommon(result);
end;

function Convert(const AValue: TConvUtilFloat;
                 const FromUnit1, FromUnit2,  ToUnit1, ToUnit2: TConvType):
                 TConvUtilFloat;
begin
  Result := Convert(Convert(AValue, FromUnit1, ToUnit1), ToUnit2, FromUnit2);
end;

function AssignFamilies(Strings: TStrings): integer;
var
  i: integer;
begin
  strings.clear;
  for i := 0 to length(Families)-1 do begin
    if Families[i] <> '' then begin
      strings.addobject(ConvFamilyToDescription(i), TObject(ptrint(i)));
    end;
  end;
  result := Strings.Count;
end;

function AssignConvTypes(aFamily: TConvFamily; Strings: TStrings): integer;
var
  i: Integer;
begin
  strings.clear;
  for i := 0 to Length(Conversions) - 1 do begin
    if Assigned(Conversions[i]) and (Conversions[i].ConvFamily = AFamily) then begin
      strings.addObject(ConvTypeToDescription(i), TObject(ptrint(i)));
    end;
  end;
  result := Strings.Count;
end;

finalization
  UnregisterAllConversionFamilies;
end.

