#include <ESP8266WiFi.h>
#include <WiFiClient.h>
//#include <ESP8266Ping.h>
#include <ESP8266httpUpdate.h>
#include "support.h"
#include "monitor.h"
#include "logging.h"

extern "C" {
#include "user_interface.h"
}

// states DISABLED, MONITORING, CYCLING, SPINNING

// Testing various timing patterns
//
//                          blinks
//                          |    blinkofftime
//                          |    |     blinkontime
//                          |    |     |      offtime       
//                          |    |     |      |
uint16_t blink_params[] = { 1,   0,   35,    35,  // PANIC_PATTERN
                            2, 200,   30 , 1740,  // HEARTBEAT           - monitoring 
                            1,   0,  500,   500,  // ROUTER_CYCLING 
                            4, 100,   50,  2500,  // WAIT_PATTERN
                            1,   0, 1980,    20,  // ROUTER_ON_PATTERN   - monitoring disabled, router on
                            1,   0,   20,  1980,  // ROUTER_OFF_PATTERN  - monitoring disabled, router off
                            6, 250,   50,  2500,  // POWER_UP_PATTERN
                            0,   0,    0,     0}; // OFF_PATTERN  
                            /*
                            0,   0,    0,     1,  // ON_PATTERN
                            */
                            
mdBlinky blinky(LED_PIN, ACTIVATE_LED);  

int blinkPattern = -1;

void setupBlinky(void) {
  blinky.setParams(blink_params, POWER_UP_PATTERN);
  blinkPattern = POWER_UP_PATTERN;
}

void blinkyUpdate(void) {
  blinky.blink();
}

void setBlinkyPattern(int pattern) {
  if (pattern != blinkPattern) {
    blinkPattern = pattern;
    blinky.setParams(blink_params, pattern);
    //dbg sendToLogf(LOG_DEBUG, "DBG: setBlinkyPattern(%d)", pattern);
  }  
}

void setupRouter(void) {
  pinMode(RELAY_PIN, OUTPUT);
  digitalWrite(RELAY_PIN, ACTIVATE_RELAY); 
}

boolean isRouterOn(void) {
  return (digitalRead(RELAY_PIN) == ACTIVATE_RELAY);
}

boolean isRouterOff(void) {
  return !isRouterOn();
}

void setRouter(int value) {
  digitalWrite(RELAY_PIN, value);
  sendToLogP(LOG_INFO, (value == ACTIVATE_RELAY) ? PSTR("Turning router on") : PSTR("Turning router off"));
}

void setRouterOn(void) {
  setRouter(ACTIVATE_RELAY);
}

void setRouterOff(void) {
  setRouter(1-ACTIVATE_RELAY);
}
  
int toggleRouter(void) {
  if (isRouterOn()) {
    setRouterOff();
  } else {
    setRouterOn();
  }
  return isRouterOn();
}  


// delay that also blinks LED
//
void localDelay(uint32_t ms) {
  unsigned long time = millis();
  while ((millis() - time) < ms) {
    yield();
    blinkyUpdate();
  }
}

void checkCleanBoot(void) {
  if (isUartBoot()) {
    sendToLogP(LOG_INFO, PSTR("restart after UART upload, power device off and on now!"));
    setBlinkyPattern(PANIC_PATTERN);
    while (1) {
      blinky.blink();
      yield();
    }
  }
}

boolean getConnectResult(void) {
  unsigned long start = millis();
  while (millis() - start < config.connectTime) {
    yield();
    blinkyUpdate();
    switch (WiFi.status()) {
      case WL_CONNECTED:
        sendToLogPf(LOG_INFO, PSTR("Connected to WiFi network %s as %s at %s"), WiFi.SSID().c_str(), WiFi.hostname().c_str(), WiFi.localIP().toString().c_str());
        return true;
        break;
        
      case WL_NO_SSID_AVAIL: 
        sendToLogPf(LOG_ERR, PSTR("Network %s not found"),  WiFi.SSID().c_str());
        return false;
        break;
        
      case WL_CONNECT_FAILED: 
        sendToLogP(LOG_ERR, PSTR("Incorrect password")); 
        return false;
        break;
    }
  }
  sendToLogP(LOG_ERR, PSTR("Network connection attempt timed out"));
  return false;
}  

/*
typedef enum WiFiMode 
{
    WIFI_OFF = 0, 
    WIFI_STA = 1, 
    WIFI_AP = 2, 
    WIFI_AP_STA = 3
} WiFiMode_t;
*/
static const char *wifiModeS[] = {
  "OFF",       // WIFI_OFF    = 0 = 0b00
  "STA",       // WIFI_STA    = 1 = 0b01
  "AP",        // WIFI_AP     = 2 = 0b10
  "AP & STA"}; // WIFI_AP_STA = 3 = 0b11

bool setWiFiMode(WiFiMode_t mode) {
  WiFiMode_t currentMode = WiFi.getMode();
  bool result = currentMode == mode;
  if (!result) {
    if (!currentMode) {
     // if currentMode != WIFI_OFF, make sure WiFi is really active.
     // See: https://github.com/esp8266/Arduino/issues/6172#issuecomment-500457407
     // Thank you ESPEasy (ESPEasyWifi.ino setWifiMode() )
     WiFi.forceSleepWake(); 
     delay(100);
    }       
    result = WiFi.mode(mode);
    if (result) {
      if (mode == WIFI_OFF) {
        delay(1000);
        WiFi.forceSleepBegin();
      } else {
        delay(30);
      }
    }
  }
  if (result) {
    //sendToLogPf(LOG_DEBUG, PSTR("WiFi.mode set to %s"), wifiModeS[WiFi.getMode()], wifiModeS[mode]);
    sendToLogPf(LOG_INFO, PSTR("WiFi.mode set to %s"), wifiModeS[WiFi.getMode()]);
  } else {
    sendToLogPf(LOG_ERR, PSTR("WiFi.mode is %s, should be %s"), wifiModeS[WiFi.getMode()], wifiModeS[mode]);
  }
  return result;  
}

bool checkWiFiMode(WiFiMode_t mode, char * functionName) {
  bool result = ((mode == WIFI_STA) || (mode == WIFI_AP));
  if (!result) sendToLogPf(LOG_ERR, PSTR("%d is invalid parameter for %s"), mode, functionName);
  return result;
}

bool startWiFiMode(WiFiMode_t mode) {
  if (checkWiFiMode(mode, "startWiFiMode")) {
    return setWiFiMode((WiFiMode_t) (WiFi.getMode() | mode));
  } else {
    return false;
  }
}    

bool stopWiFiMode(WiFiMode_t mode) {
  if (checkWiFiMode(mode, "startWiFiMode")) {
    return setWiFiMode((WiFiMode_t) (WiFi.getMode() ^ mode));
  } else {
    return false;
  }  
}  

void setWiFiStationConfig(void) {
  IPAddress staticIP;
  if (strlen(config.staIP)) {
    IPAddress gateway;
    IPAddress subnet;
    staticIP.fromString(config.staIP);
    gateway.fromString(config.staGateway);
    subnet.fromString(config.staMask);
    WiFi.config(staticIP, gateway, subnet);
    sendToLogPf(LOG_INFO, PSTR("Set static client IP %s, gateway %s, subnet %s"), staticIP.toString().c_str(), gateway.toString().c_str(), subnet.toString().c_str()); 
    //sendToLogPf(LOG_DEBUG, PSTR("WiFi static IP %s, gateway %s, subnet %s"), WiFi.localIP().toString().c_str(),  WiFi.gatewayIP().toString().c_str(), WiFi.subnetMask().toString().c_str());
  } else {
    staticIP.fromString("0.0.0.0");
    WiFi.config(staticIP, staticIP, staticIP);
    sendToLogP(LOG_INFO, PSTR("Client IP, gateway and subnet assigned dynamically by monitored network (DHCP)"));
  }
  localDelay(100);
}

boolean connectWiFi(char * ssid, char * pswd) {
  if (strlen(ssid) == 0) {
    sendToLogP(LOG_ERR, PSTR("Network SSID not defined"));
    return false;
  }

  if (WiFi.status() == WL_CONNECTED) {
    // already connected, that's ok if ssid = connected network name
    if (!strcmp(ssid, WiFi.SSID().c_str()))  {
      return getConnectResult(); // cheap way to log connection
    }   
    sendToLogPf(LOG_INFO, PSTR("Disconnecting from \"%s\""), WiFi.SSID().c_str() );
    WiFi.disconnect();
    localDelay(100);
    //sendToLogf(LOG_ERR, "DBG: WiFi.isConnected(): %s, WiFi.hostname: %s", (WiFi.isConnected()) ? "true" : "false", WiFi.hostname().c_str() );
  }
  
  WiFi.begin(ssid, pswd);
  return getConnectResult();
}



boolean setHostName(void) {
  bool result = false;
  if (startWiFiMode(WIFI_STA)) {
    //sendToLogP(LOG_DEBUG, PSTR("setHostName WiFi mode: %s"), wifiModeS[WiFi.getMode()]);
    if (strlen(HOST_NAME)) { 
      for (int i=0; i < 5; i++) {
        result = WiFi.hostname(config.hostname); 
        if (result) break;
        localDelay(100);
      }
    }
  }
  if (result) {
    sendToLogPf(LOG_INFO, PSTR("Hostname set to %s"), WiFi.hostname().c_str());
  } else {
    sendToLogP(LOG_ERR, PSTR("Failed to set host name"));
  }
  return result;
}  

  
void initWiFiStation(void) {
  setHostName();  // will set WiFi in WIFI_STA mode
  WiFi.setAutoConnect(true);
  WiFi.setAutoReconnect(true);
}

int pingHost(const char * host, int count) {
  if (!Ping.ping(host, count)) {
    return -1;
  } else {
    return Ping.averageTime();
  }  
}

void doRestart(restartData_t data) {
  sendToLogP(LOG_INFO, PSTR("Rebooting..."));
  //ESP.restart();
  //userRestart(data);
  userReset(data);
}


boolean doOtaUpdate(const char *url) {

  mqtt_disconnect();

  //unsigned long atime = millis();
  int oldPattern = blinkPattern;
  //setBlinkyPattern(PANIC_PATTERN);
  
  sendToLogP(LOG_INFO, PSTR("Attempting OTA update"));

  // assume it won't work
  boolean updated = false;
  boolean otaNetConnected = false;
  
  // save current station credetials
  char currentSsid[SSID_SZ] = {0};
  char currentPsk[PWD_SZ] = {0};

  if (WiFi.SSID().length()) {
    strlcpy(currentSsid, WiFi.SSID().c_str(), SSID_SZ);
    strlcpy(currentPsk, WiFi.psk().c_str(), PWD_SZ);
  }

  
  // Decide on which network to connect
  if ( (strlen(config.otaSsid)) && (strcmp(currentSsid, config.otaSsid)) )   {
    sendToLogPf(LOG_INFO, PSTR("Connecting with the OTA WiFi network %s"), config.otaSsid);
    otaNetConnected = connectWiFi(config.otaSsid, config.otaPsk);
  } else {
    otaNetConnected = WiFi.isConnected();
    if (!otaNetConnected) sendToLogP(LOG_ERR, "Not connected to OTA Wi-Fi network");
  }

  if (otaNetConnected) { 
  
    sendToLogPf(LOG_INFO, PSTR("OTA update URL: %s"), url);
  
    //while ((millis() - atime) < 3000) {
    //  yield();
    //  blinkyUpdate();
    // }
    // setBlinkyPattern(OFF_PATTERN);
      
    ESPhttpUpdate.rebootOnUpdate(false);
    
    WiFiClient OtaClient;
    int res = ESPhttpUpdate.update(OtaClient, url);
    delay(5);
    //int res = ESPhttpUpdate.update(url);
    //delay(10);
    Serial.printf("ESPhttpUpdate.update(OtaClient, url) -> %d \n", res);
    switch(res) {  
      case HTTP_UPDATE_FAILED:
        sendToLogPf(LOG_ERR, PSTR("OTA update failed. Error (%d): %s"), ESPhttpUpdate.getLastError(), ESPhttpUpdate.getLastErrorString().c_str());
        break;
    
      case HTTP_UPDATE_NO_UPDATES:
        sendToLogP(LOG_ERR, PSTR("OTA update failed. Error: No updates"));
        break;
    
      case HTTP_UPDATE_OK:
        sendToLogP(LOG_INFO, PSTR("OTA update successful. Restarting..."));
        updated = true;
        break;
    }
     
    // if ota was done on network config.otaSsid and it is different from the monitored 
    // WiFi network, then the latter will have to be manually setup again
    if (strcmp(config.otaSsid, currentSsid)) {
      
      connectWiFi(currentSsid, currentPsk);
    }
  }
  
  setBlinkyPattern(oldPattern);
  return updated;
}   


void updateOta(const char *url) {
  if (doOtaUpdate(url)) doRestart(USER_RESTART_OTA_CMD);
}

const static char *reasons[] = {
  "power on",
  "hardware watchdog timed out",
  "software exception %d occured",
  "software watchdog timed out",
  "software restart() or reset() invoked",
  "ESP woke from deep sleep",
  "external system reset"
};

void restartCheck(void) {
  char msg[MSG_SZ] = {0};
  boolean updateNeeded = false;
  restartReason_t restartReason;
  restartData_t restartData;
  restartCount_t restartCount;
  
  restartReason = getRestartReason(restartCount, restartData);

  if  ( (restartReason == REASON_DEFAULT_RST) || (restartReason == REASON_EXT_SYS_RST) )  {
    sendToLogP(LOG_INFO, PSTR("Device turned on"));

  } else if ( restartReason >= REASON_LWD_RST ) {
    updateNeeded = true;
    if (restartData < MODULE_COUNT-2) {
      snprintf_P(msg, sizeof(msg), PSTR("%sModule()"), modules[restartData]);
    } else if (restartData == LOOP_END) {
      strlcpy(msg, modules[MODULE_COUNT-1], sizeof(msg));
    } else {
      strlcpy(msg, modules[MODULE_COUNT-2], sizeof(msg));
    }
    switch(restartReason) {
      case REASON_LWD_RST: sendToLogPf(LOG_INFO, PSTR("Loop watchdog reset at %s"), msg); break;
      case REASON_LWD_LOOP_RST: sendToLogPf(LOG_INFO, PSTR("Loop watchdog, loop() not completed, last module %s"), msg); break;
      case REASON_LWD_OVW_RST: sendToLogPf(LOG_INFO, PSTR("Loop watchdog overwritten, last module %s"), msg); break;     
    }
    
  } else if (restartReason == REASON_USER_RESTART) {
      sendToLogPf(LOG_DEBUG, PSTR("User initiated restart (data: %d)"), restartData);
      
  } else if (restartReason == REASON_USER_RESET) {
      sendToLogPf(LOG_DEBUG, PSTR("User initiated reset (data: %d)"), restartData);
      
  } else if (restartReason == REASON_EXCEPTION_RST) {
    updateNeeded = true;
    sendToLogf(LOG_DEBUG, reasons[REASON_EXCEPTION_RST], restartData);
    
  } else {
    sendToLogPf(LOG_DEBUG, PSTR("Reason for restart: %s"), reasons[restartReason]);
  }

  sendToLogPf(LOG_DEBUG, PSTR("Count of successive restarts for that reason: %d"), restartCount);
  
  if ( updateNeeded && (restartCount >= AUTO_UPDATE_COUNT) && (AUTO_UPDATE_COUNT > 0) ) {
    if (doOtaUpdate(config.autoUrl)) {
      doRestart(USER_RESTART_OTA_LWDT);
    }  
  }
}
