/*
 * Simple router monitor
 * 
 */

#include <ESP8266WiFi.h>
#include <WiFiClient.h>
#include <PubSubClient.h>

#include <mdEspRestart.h>
#include <mdButton.h>
#include <mdBlinky.h>

#include "config.h"
#include "monitor.h"
#include "support.h"
#include "commands.h"
#include "logging.h"

extern "C" {
#include "user_interface.h"
}

//#define TEST_AUTO_OTA

WiFiClient    espClient;
PubSubClient  mqttClient(espClient);

// loop watchdog

#define BUTTON_MODULE     0
#define LED_MODULE        1
#define INPUT_MODULE      2
#define NET_MODULE        3
#define MQTT_MODULE       4

static const char *modules[] = {
  "button",
  "led",
  "input",
  "net",
  "mqtt",
  "unknown",
  "end of loop"
};

#define MODULE_COUNT (sizeof (modules) / sizeof (const char *))

/*
 * MODULES
 */

mdButton button(BUTTON_PIN);
mdButton button2(BUTTON2_PIN);


// buttonModule monitors the push button and
// performs actions in response to user entered
// commands. 
//
void buttonModule(void) {
  lwdtStamp(BUTTON_MODULE);  
  switch (button.status()) {
    case -1: doRestart(USER_RESTART_BTN); break;   
    case  0: break; 
    case  1: doCommand(FROM_BTN, "router toggle"); break;
    case  2: (monitorState == DISABLED) ? doCommand(FROM_BTN, "monitor on") : doCommand(FROM_BTN, "monitor off"); break;
    case  3: break;
    default: doCommand(FROM_BTN, "update");  break;
  }  

  switch (button2.status()) {
    case -1: doRestart(USER_RESTART_BTN); break;   
    case  0: break; 
    case  1: doCommand(FROM_BTN, "router toggle"); break;
    case  2: (monitorState == DISABLED) ? doCommand(FROM_BTN, "monitor on") : doCommand(FROM_BTN, "monitor off"); break;
    case  3: break;
    default: doCommand(FROM_BTN, "update");  break;
  }  
}

// inputModule pulls any available char from the serial Rx line and adds it to a buffer.
// When a end of line character ('\n' or CR 0x0D) of character is read, the string is
// send to the command interpreter and the buffer is cleared. This is usefull during
// development but not very useful in practice.
//
void inputModule() {
  
  static String inputString;
  boolean stringComplete = false;

  lwdtStamp(INPUT_MODULE);

  // while (1) { yield(); } //to test lwd watchdog
  
  while (Serial.available() && !stringComplete) {
    char inChar = (char)Serial.read();
    if (inChar == '\n') {
      stringComplete = true;
    } else {
      inputString += inChar;
    }
  }
  if (stringComplete) {
    doCommand(FROM_SERIAL, inputString.c_str());
    inputString = "";
    stringComplete = false;
  }  
}


// ledModule calls on the LED blinky object defined in support.ino to
// update the LED state
//
void ledModule(void) {
  lwdtStamp(LED_MODULE);
  blinkyUpdate();
}


// monitorModule calls on the state machine in monitor.ino to
// update its state
//
void monitorModule(void) {
  lwdtStamp(NET_MODULE);
  monitorUpdate();
}

#ifdef TEST_AUTO_OTA
unsigned long exceptionTime;
#endif


unsigned long mqttTime; 
String mqttCommand;

//  This function will be called by mqttClient whenever it
//  receives a subscribed MQTT message from the broker
//  with the topic "modem/command"
//
void mqtt_callback(char* topic, byte* data, unsigned int length) {
  char payload[length + 1];
  memcpy(payload, data, sizeof(payload));
  payload[sizeof(payload) - 1] = 0;
  mqttCommand = payload;
  sendToLogPf(LOG_DEBUG, PSTR("MQTT [%s]: %s)"), topic, payload);
}

void setup() {
  setupRouter(); // turns router on
  Serial.begin(115200);
  delay(10);
  Serial.println("\n\n");
  checkCleanBoot();
  setupBlinky();
  sendToLogPf(LOG_INFO, PSTR("Network Monitor (version %d.%d.%d)"), (VERSION >> 16) & 0xFF, (VERSION >> 8) & 0xFF, VERSION & 0xFF);
  loadConfig();
  initWiFiStation();
  setWiFiStationConfig();
  connectWiFi(config.netSsid, config.netPsk);
  restartCheck();

  if (strlen(config.mqttHost) > 0) {
    mqttClient.setServer(config.mqttHost, config.mqttPort);
    mqttClient.setCallback(mqtt_callback);  
  } else {  
    sendToLogPf(LOG_ERR, "mqtt host not defined");
  }  
  mqttTime = millis() - MQTT_INTERVAL;

  lwdtInit(LOOP_TIMEOUT);
  if (WiFi.status() == WL_CONNECTED) {
    enableMonitor();
  } else {
    startSpinning();  
  }
  sendToLogP(LOG_DEBUG, PSTR("Setup completed"));
  
#ifdef TEST_AUTO_OTA
  exceptionTime = millis();
}  

int exceptres; 
#else
}
#endif  

void mqtt_disconnect(void) {
  //sendToLogP(LOG_DEBUG, "mqtt_disconnect");
  if (mqttClient.connected()) {
    sendToLogP(LOG_INFO, "Disconnecting mqtt");
    mqttClient.disconnect(); 
    delay(10);
  }
}
  
void mqtt_reconnect() {
  char topic[MSG_SZ];
  char msg[MSG_SZ];
  //sendToLogP(LOG_DEBUG, PSTR("mqtt_reconnect"));
  mqttTime = millis();
  if ( (!mqttClient.connected()) && (WiFi.status() == WL_CONNECTED)) {
    sendToLogP(LOG_INFO, PSTR("Reconnecting to mqtt broker"));
    if (mqttClient.connect(config.hostname)) {
      sendToLogPf(LOG_INFO, PSTR("Connected to mqtt broker as %s"), config.hostname);
      snprintf(topic, sizeof(topic), "%s/%s/#", config.hostname, config.mqttCommand);
      sendToLogPf(LOG_INFO, PSTR("Subscribing to mqtt topic \"%s\""), topic);
      mqttClient.subscribe(topic);
      mqttClient.loop();
      delay(10);
      // send first message
      //snprintf(topic, sizeof(topic), "%s/%s/#", config.hostname, config.mqttResponse);
      //snprintf(msg, sizeof(msg), "connected, firmware version %d.%d.%d", (VERSION >> 16) & 0xFF, (VERSION >> 8) & 0xFF, VERSION & 0xFF);
      //mqttClient.publish(topic, msg);
      //delay(100);
      //Serial.printf("mqttClient.connected(): %s\n", (mqttClient.connected()) ? "yes" : "no");
    } else {
      sendToLogPf(LOG_ERR, PSTR("Connection to mqtt broker failed, rc=%d"), mqttClient.state());
    }
  } 
}  

void mqttModule() {
  lwdtStamp(MQTT_MODULE); 
  mqttClient.loop();
  if (mqttCommand.length() > 0) {
    doCommand(FROM_MQTT, mqttCommand.c_str());
    mqttCommand = "";
    mqttClient.loop();
  }
  if ((!mqttClient.connected()) &&  (millis() - mqttTime > config.mqttInterval)) {
    mqtt_reconnect();
  }
}

void loop() {
  lwdtFeed();
  mqttModule();
  buttonModule();
  ledModule();
  monitorModule();
  inputModule();
#ifdef TEST_AUTO_OTA  
  if (millis() - exceptionTime > 60*1000) {
    exceptres = 5 / (2 - 2); 
  }
#endif  
  lwdtStamp(); 
}
