#include "logging.h"
#include <ESP8266WiFi.h>
#include <WiFiUdp.h>
#include "config.h"

extern "C" {
#include "user_interface.h"
}

void mstostr(unsigned long milli, char* sbuf, int sbufsize) {
  int sec = milli / 1000;
  int min = sec / 60;
  int hr = min / 60;
  min = min % 60;
  sec = sec % 60;
  int frac = (milli % 1000);
  snprintf_P(sbuf, sbufsize-1, PSTR("%02d:%02d:%02d.%04d"), hr,min,sec,frac);
}

#define LOGSZ 520

WiFiUDP udp;


/*
 * A string sent to the log will be 
 *  a) printed to the serial if level has a higher or equal priority to serialThreshold
 *  b) sent to the mqtt broker if level has a higher or equal priority to mqttThreshold 
 *  c) sent to the syslog server if level has a higher or equal priority to syslogThreshold 
 */

void sendToLog(uint8_t level, const char *line ) {
  //Serial.printf("\nSendToLog(%d, %s)\n", level, line);
  String message;
  char mxtime[15];
  mstostr(millis(), mxtime, sizeof(mxtime));
  message = String(mxtime) + " " + String(line);

  if (level <= config.logLevelUart) {
    Serial.printf("%s\n", message.c_str());
  }

  //Serial.printf((!mqttClient.connected()) ? "mqttClient NOT connected\n" : "");
  if ( (level <= config.logLevelMqtt) && mqttClient.connected() )  {
    String topic = String(config.hostname) + "/" + String(config.mqttResponse);
    //Serial.printf("mqtt sending [%s]:  %s\n", topic.c_str(), message.c_str());
    mqttClient.publish(topic.c_str(), message.c_str());
  }
  
  if ((level <= config.logLevelSyslog) && (WiFi.status() == WL_CONNECTED)) {
    message = String(config.hostname) + ": " + String(line);
    if (udp.beginPacket(config.syslogHost, config.syslogPort)) {
      udp.write(message.c_str());
      udp.endPacket();
      delay(1);  // Add time for UDP handling (#5512)
    }    
  }
}

void sendToLogf(uint8_t level, const char *format, ...) {
  va_list args;
  char line[250];
  va_start(args, format);
  vsnprintf(line, 250, format, args);
  va_end(args);
  sendToLog(level, line);
}

void sendToLogP(uint8_t level, const char *line) {
  char msg[LOGSZ];
  strcpy_P(msg, line);
  sendToLog(level, msg);
}  

void sendToLogP(uint8_t level, const char *linep1, const char *linep2) {
  char msg[LOGSZ];
  char msg2[LOGSZ];
  strcpy_P(msg, linep1);
  strcpy_P(msg, linep2);
  strncat(msg, msg2, sizeof(msg));
  sendToLog(level, msg);
}

void sendToLogPf(uint8_t level, const char *pline, ...) {
  va_list args;
  char msg[LOGSZ];
  char msg2[LOGSZ];
  strcpy_P(msg, pline);
  va_start(args, pline);
  vsnprintf(msg2, sizeof(msg2), msg, args);
  va_end(args);
  sendToLog(level, msg2);
}

#ifdef DUMP_WEB_LOG
void dumpWebLog(void) {
  if (logcount) {
    Serial.println("Log: {");
    int j = (logcount < MAX_LOG_LINES) ? logidx-logcount : logidx;
    for (int i = 0; i < logcount; i++) {
      if (j == MAX_LOG_LINES) j = 0;
      Serial.printf("  %s\n", Log[j].c_str());
      j++;
    }  
    Serial.println("}");
  } else {
    Serial.println("Log: {empty}");
  }  
} 
#endif
 
