#include <EEPROM.h>
#include "config.h"
#include "logging.h"

#define MAX_EEPROM_SIZE 4096

/*
 * https://esp8266.github.io/Arduino/versions/2.0.0/doc/libraries.html
 * 
 * This is a bit different from standard EEPROM class. You need to call EEPROM.begin(size) 
 * before you start reading or writing, size being the number of bytes you want to use. 
 * 
 * Size can be anywhere between 4 and 4096 bytes.
 * 
 * EEPROM.write does not write to flash immediately, instead you must call EEPROM.commit() 
 * whenever you wish to save changes to flash. EEPROM.end() will also commit, and will 
 * release the RAM copy of EEPROM contents.
 * 
 * EEPROM library uses one sector of flash located just after the SPIFFS.
 * 
 * Looking at EEPROM.write code in https://github.com/esp8266/Arduino/blob/master/libraries/EEPROM/EEPROM.cpp
 * it seems clear that a lazy write algorithm is used in EEPROM.commit(), meaning a ram copy of 
 * the content of the EEPROM is used as a buffer and only if changes are made to that copy with 
 * EEPROM.write will there be a change.
 */

void useDefaultConfig(void) {
  memset(&config, 0x00, sizeof(config_t));
  config.version = CONFIG_VERSION;  
  config.wifiDownInterval = WIFI_DOWN_INTERVAL;  
  config.internetLostInterval = INTERNET_LOST_INTERVAL;  
  config.intervalBetweenPings = INTERVAL_BETWEEN_PINGS;  
  config.waitAfterWifiDown = WAIT_AFTER_WIFI_DOWN;  
  config.waitAfterInternetLost = WAIT_AFTER_INTERNET_LOST;  
  config.routerOffInterval = ROUTER_OFF_INTERVAL;  
  config.connectTime = CONNECT_TIME;
  config.mqttInterval = MQTT_INTERVAL;
  
  strlcpy(config.hostname, HOST_NAME, HOST_NAME_SZ);
  strlcpy(config.netSsid, NET_SSID, SSID_SZ);
  strlcpy(config.netPsk, NET_PSK, PWD_SZ);
  strlcpy(config.staIP, STA_IP, IP_SZ);
  strlcpy(config.staGateway, STA_GATEWAY, IP_SZ);
  strlcpy(config.staMask, STA_MASK, IP_SZ);

  strlcpy(config.mqttHost, MQTT_HOST, SSID_SZ);
  config.mqttPort = MQTT_PORT;
  strlcpy(config.mqttCommand, MQTT_COMMAND, TOPIC_SZ);
  strlcpy(config.mqttResponse, MQTT_RESPONSE, TOPIC_SZ);

  strlcpy(config.syslogHost, SYSLOG_HOST, SSID_SZ);
  config.syslogPort = SYSLOG_PORT;

  strlcpy(config.otaSsid, OTA_SSID, SSID_SZ);
  strlcpy(config.otaPsk, OTA_PSK, PWD_SZ);
  strlcpy(config.otaUrl, OTA_URL, URL_SZ);
  strlcpy(config.autoUrl, AUTO_URL, URL_SZ);

  strlcpy(config.targets[0], PING_TARGET_0, URL_SZ);
  strlcpy(config.targets[1], PING_TARGET_1, URL_SZ);
  strlcpy(config.targets[2], PING_TARGET_2, URL_SZ);

  config.logLevelUart = LOG_LEVEL_UART;
  config.logLevelMqtt = LOG_LEVEL_MQTT;
  config.logLevelSyslog = LOG_LEVEL_SYSLOG;

  sendToLogPf(LOG_INFO, PSTR("Using default configuration (version %d)"), config.version);
}

void getConfigVersion(void){
  sendToLogPf(LOG_INFO, PSTR("Current configuration version %d"), config.version);
}

// save config to flash memory
//
void saveConfigToEEPROM(void) {
  EEPROM.begin(sizeof(config_t));
  EEPROM.put(0, config);
  EEPROM.end();
  eepromConfigHash = getConfigHash();
  sendToLogPf(LOG_INFO, PSTR("Saved current configuration to flash memory (version %d)"), config.version);
}

uint32_t getConfigHash() {
  uint32_t hash = 0;
  uint8_t *bytes = (uint8_t*)&config;

  for (uint16_t i = 0; i < sizeof(config_t); i++) hash += bytes[i]*(i+1);
  return hash;
}


// Copies configuration saved in EEPROM to config buffer and calculates
// the hash for the config buffer. If the EEPROM saved configuration is 
// an older version, if will be shorter than the current config size
// and garbage (most likely 0xFF) will be at the end. 
//
void loadConfigFromEEPROM(void) {
  memset(&config, 0x00, sizeof(config_t));
  EEPROM.begin(sizeof(config_t));
  EEPROM.get(0, config);
  EEPROM.end();
  eepromConfigHash = getConfigHash();
  sendToLogPf(LOG_INFO, PSTR("Loaded current configuration from flash memory (version %d)"), config.version);
}


void loadConfig(void)
{
  loadConfigFromEEPROM();
  if (config.version != CONFIG_VERSION ) {  
    useDefaultConfig();
    eepromConfigHash = 0; // will mark configuration for automatic saving 
  }   

#ifdef CONFIG_VERSIONING
   
  if ((config.version > CONFIG_VERSION ) || (!config.version) )
  {  
    useDefaultConfig();
    eepromConfigHash = 0;
  }   
  else if (config.version < CONFIG_VERSION) 
  {
    // add missing default configurations. This will overwrite garbage 
    char msg[MSG_SZ];
    snprintf(msg, MSG_SZ, "Updated current configuration (version %d) to version %d", config.version, CONFIG_VERSION);
    sendToLog(LOG_INFO, msg); 
    
    config.version = CONFIG_VERSION;
    saveConfigToEEPROM();
  }
#endif
}


void clearEEPROM(void)
{
  EEPROM.begin(sizeof(word));
  for (int i = 0 ; i < sizeof(word) ; i++) {
    EEPROM.write(i, 0xFF);
  }
  EEPROM.end();  
  sendToLogP(LOG_INFO, PSTR("Cleared configuration in EEPROM")); 
}


#ifdef DUMP_CONFIG

void dumpConfig(void)
{ 
  #define CFG_COLS 16
  uint16_t i, col;
  uint8_t *buffer = (uint8_t *) &config;
  char ascii[19];
  i = 0;
  col = 0;
  
  Serial.printf("\nconfig(%d bytes):", sizeof(config_t));
  for (i = 0; i < sizeof(config_t); i++) 
  {
    if (col == 0)  Serial.printf("\n%04X: ", i);
    Serial.printf("%02X ", buffer[i]);
    if (buffer[i] < 0x20)
    {
      ascii[col] = ' ';
    }  
    else if (buffer[i] > 127)
    {
      ascii[col] = '.';
    }
    else
    {
      ascii[col] = buffer[i];
    }  
    ascii[col+1] = 0; 
    if (++col >= CFG_COLS)
    {
      col = 0;
      Serial.printf("  %s", ascii);
    }
  }  
  if (col > 0) Serial.printf("  %s\n", ascii);
}

void dumpEEPROMConfig(void)
{ 
  if (!eepromOverflow(sizeof(config_t)))
  {
    #define CFG_COLS 16
    uint16_t i, col;
    char ascii[19];
    char c;
    i = 0;
    col = 0;
    Serial.printf("\nEEPROM config(%d bytes):", sizeof(config_t));
    EEPROM.begin(sizeof(config_t));
    for (i = 0; i < sizeof(config_t); i++) 
    {
      if (col == 0)  Serial.printf("\n%04X: ", i);
      c = EEPROM.read(i);
      Serial.printf("%02X ", c);
      if (c < 0x20)
      {
        ascii[col] = ' ';
      }  
      else if (c > 127)
      {
        ascii[col] = '.';
      }
      else
      {
        ascii[col] = c;
      }  
      ascii[col+1] = 0; 
      if (++col >= CFG_COLS)
      {
        col = 0;
        Serial.printf("  %s", ascii);
      }
    } 
    if (col > 0) Serial.printf("  %s\n", ascii);
    EEPROM.end();
  }  
}

#endif
