
#ifndef _CONFIG_H_
#define _CONFIG_H_


// *** Default time periods ***

#define WIFI_DOWN_INTERVAL         30000 //  30 seconds
#define INTERVAL_BETWEEN_PINGS    180000 // 180 seconds =  3 minutes
#define INTERNET_LOST_INTERVAL    720000 // 720 seconds = 12 minutes (at least 3 failed pings)
#define WAIT_AFTER_WIFI_DOWN      300000 // 300 seconds =  5 minutes
#define WAIT_AFTER_INTERNET_LOST  900000 // 900 seconds = 15 minutes
#define ROUTER_OFF_INTERVAL        10000 //  10 seconds
#define CONNECT_TIME              120000 // 120 seconds =  2 minutes
#define MQTT_INTERVAL            60*1000 //  60 seconds =  1 minute

// *** Default hostname of device on WiFi network  ***
// 
// Allowed chars: "a".."z", "A".."Z", "0".."9" "-", but must not start nor end with "-". 
// Assume that the name is not case sensitive. Ref: https://en.wikipedia.org/wiki/Hostname

#define HOST_NAME    "routermon-1"

// *** Monitored WiFi network credentials ***

#define NET_SSID    ""   // must be defined  
#define NET_PSK     ""   // must be defined
#define STA_IP      ""   // leave as "" for DHCP
#define STA_GATEWAY ""
#define STA_MASK    ""

// *** MQTT broker *** 

#define MQTT_HOST     "192.168.1.22"  // must be defined
#define MQTT_PORT     1883
#define MQTT_COMMAND  "command"
#define MQTT_RESPONSE "response"

// *** Syslog server ***

#define SYSLOG_HOST "192.168.1.22"   // must be defined
#define SYSLOG_PORT 514


// *** OTA webserver WiFi network, binary file URLs ***

#define OTA_SSID ""      // if empty, then the monitored network 
#define OTA_PSK  ""      // credentials will be used
#define OTA_URL  "http://192.168.1.22/router/routermon.ino.bin"  // used when update command is executed
#define AUTO_URL "http://192.168.1.22/router/routermon.good.bin" // used when auto update in restart loop is executed
#define AUTO_UPDATE_COUNT  3      // number of consecutive watchdog timeouts and exceptions before autoUpdating firmware, 0 disables;

// *** Ping targets ***

#define PING_TARGET_0 "google.com" 
#define PING_TARGET_1 "bing.com"
#define PING_TARGET_2 "amazon.com"


// *** Log levels ***

#define LOG_LEVEL_UART LOG_INFO
#define LOG_LEVEL_MQTT LOG_INFO
#define LOG_LEVEL_SYSLOG LOG_ERR

//==========//==========//==========//==========//==========//==========//==========//
// Everything above are default values that can be modified with commands
// Everything below should be set once for all and cannot be modified at run time.
//==========//==========//==========//==========//==========//==========//==========//

#define VERSION 0x000204  // version 0.2.4

//  *** I/O pins ***

#define BUTTON_PIN     14           // The Sonoff Basic R1 free pin on the header is GPIO 14 ( D5 on NodeMCU/D1 mini) Active low 
#define BUTTON2_PIN     0           // The Sonoff Basic tactile switch is on GPIO 0 (D3 on NodeMCU/D1 mini)
#define LED_PIN        13           // The Sonoff Basic LED is on GPIO 13 ( D7 on NodeMCU/D1 mini)
#define ACTIVATE_LED   ACTIVE_LOW   // The Sonoff Basic LED is turned on with a LOW to the pin
#define RELAY_PIN      12           // The Sonoff Basic relay is on GPIO 13 (D6 on NodeMCU/D1 mini)
#define ACTIVATE_RELAY ACTIVE_HIGH  // The Sonoff Basic relay is turned on with a HIGH to the pin

#define PING_TARGET_COUNT 3 

#define LOOP_TIMEOUT      240000 // 240 seconds = 4 minute before loop watchdog times out if not fed


// Maximum number of characters in string including terminating 0
//
#define URL_SZ             81  
#define SSID_SZ            33  
#define IP_SZ              17  // 255.255.255.255
#define PWD_SZ             65  // minimum 8
#define HOST_NAME_SZ       32  // maximum size of 31 bytes for OpenSSL e-mail certificates
#define MSG_SZ            441  // needs to be big enough for "reach" command (i.e. > 3*URL_SZ) 
#define TOPIC_SZ       PWD_SZ    

#define CONFIG_VERSION 1

struct config_t {
  uint32_t version;               // 4 bytes will be 0 or 0xFFFFFFFF if EEPROM was erased
  uint32_t wifiDownInterval;      // time without Wi-Fi connection before declaring WiFi down
  uint32_t internetLostInterval;  // time without successful ping before declaring Internet unreachable
  uint32_t intervalBetweenPings;  // shortest time between pings
  uint32_t waitAfterWifiDown;     // time to wait after Wi-Fi down up before resuming monitoring
  uint32_t waitAfterInternetLost; // time to wait after internet unreachable before resuming monitoring
  uint32_t routerOffInterval;     // time to turn the router off when restarting it
  uint32_t connectTime;           // time to get result after WiFiConnect   
  uint32_t mqttInterval;          // time between attemps a connecting to MQTT broker

  char hostname[HOST_NAME_SZ];    // used for Wi-Fi connection and MQTT broker connection
  char netSsid[SSID_SZ];          // SSID name of Wi-Fi network to monitor
  char netPsk[PWD_SZ];            // Passkey (password) monitored Wi-Fi network
  char staIP[IP_SZ];              // Static IP or dynamic IP (DHCP if ""
  char staGateway[IP_SZ];         // Must be specified if static IP otherwise ignored
  char staMask[IP_SZ];            // Must be specified if static IP otherwise ignored

  char mqttHost[URL_SZ];          // URL of MQTT broker
  uint16_t mqttPort;              // MQTT port
  char mqttCommand[TOPIC_SZ];
  char mqttResponse[TOPIC_SZ];
  
  char syslogHost[URL_SZ];        // URL of Syslog server
  uint16_t syslogPort;            // Syslog port
  char otaSsid[SSID_SZ];          // SSID of WiFi network on which is the web server providing OTA updates for firmware
  char otaPsk[PWD_SZ];            // Passkey (password) of the OTA update web server WiFi network
  char otaUrl[URL_SZ];            // url of new OTA firmware on the web server
  char autoUrl[URL_SZ];           // url of good fallback OTA firmware on the web server
  char targets[PING_TARGET_COUNT][URL_SZ];

  uint8_t logLevelUart;
  uint8_t logLevelMqtt;
  uint8_t logLevelSyslog;
} config;

uint32_t eepromConfigHash;

#endif
