
#include "commands.h"
#include "Arduino.h"
#include "support.h"
#include "monitor.h"
#include "user_config.h"

static const char *cmdsrc[] = {
/* FROM_BTN */     "btn",
/* FROM_SERIAL */  "uart",
/* FROM_MQTT */    "mqtt"};

static const char *cmds[] = {
  "clientip",       //  0 - client ip, gateway and subnet mask on monitored network
  "config",         //  1 - mangage configuration
  "cycle",          //  2 - power cycle router
  "help",           //  3 - help
  "log",            //  4 - log level for serial, mqtt and web output
  "monitor",        //  5 - manage network monitoring
  "mqtt",           //  6 - mqtt url, port
  "name",           //  7 - hostname
  "net",            //  8 - monitored Wi-Fi network credentials
  "ota",            //  9 - ota credential
  "ping",           // 10 - perform a ping request
  "reach",          // 11 - manage ping targets for internet state check
  "restart",        // 12 - restart the device
  "router",         // 13 - manual override of router relay
  "syslog",         // 14 - syslog url, port
  "time",           // 15 - configure time intervals
  "topic",          // 16 - mqtt topics
  "update",         // 17 - manual firmware update
  "url",            // 18 - set urls for OTA updates
  "version"         // 19 - returns firmware version
};

#define NET_COMMAND    8   // do not echo password
#define OTA_COMMAND    9   // do not echo password
#define COMMAND_COUNT (int)(sizeof (cmds) / sizeof (const char *))

static const char *params[] = {
  "[auto | <ip>, <gateway>, <mask>]",     // clientip 
  "(save|load|default|erase)",            // config
  "[<wait>]",                             // cycle
  "[<command>]",                          // help
  "(uart|mqtt|syslog) [<level>]",         // log
  "[on|off]",                             // monitor
  "[<ip> [<port>]]",                      // mqtt
  "[<hostname>]",                         // name
  "[<ssid> [<password>]]",                // net
  "[default|<ssid> [<password>]]",        // ota
  "<host>",                               // ping
  "[(1|2|3) (<host>)]",                   // reach
  "",                                     // restart
  "[on|off|toggle]",                      // router
  "[<ip> [<port>]]",                      // syslog
  "(wifi|internet|ping|longwait|shortwait|cycling|connect|mqtt) [<ms>])",  // time
  "(in|out) [<topic>]",                   // topic
  "[<url>]",                              // update
  "(ota|auto) [<url>]",                   // url
  ""                                      // version
};


#define TOKENCOUNT 4
String token[TOKENCOUNT];

// breaks up the string s into tokens
int parseString(const char *s) {

  for (int i = 0; i < TOKENCOUNT; i++) {
    token[i] = "";
  }
  int ndx = 0;
  int tokCounter = 0;
  int len = 0;

  // skip leading spaces
  while (s[ndx] == ' ') ndx++;

  while (1) {
    char inChar = s[ndx];
    ndx++;
    if ( (len > 0) && ((inChar == ' ') || (inChar == 0)) ) {
      tokCounter += 1;
      len = 0;
    }
    if ( (inChar == 0) || (tokCounter >= TOKENCOUNT) ) return tokCounter;
    if (inChar != ' ') {
      token[tokCounter] += inChar;
      len++;
    } else {
      // skip extra spaces following inChar == ' '
      while (s[ndx] == ' ') ndx++;
    }
  }
}

// returns the command index of the token[idx] or -1 if not a command
int commandId(int idx, int counter) {
  if (counter > 0) {
    token[idx].toLowerCase();
    for (int i = 0; i<COMMAND_COUNT; i++) {
      if (token[idx].equals(cmds[i])) {
        /*
        Serial.print("token[");
        Serial.print(idx);
        Serial.print("](=");
        Serial.print(token[idx]);
        Serial.print(") has index ");
        Serial.println(i);
        */
        return i;
      }
    }
  }
  return -1;
}     



//      1       2     2     3        4  <<< counter
//      0       1     1     2        3  <<< index
//  "clientip [auto|<ip> <gateway> <mask>]]"
//
cmndError_t doClientIp(int counter, int &index) {
  IPAddress ipa;
  bool restartClient = false;
  if (counter == 2) {
    token[1].toLowerCase();
    if (token[1].equals("auto")) {
      restartClient = (strlen(config.staIP) > 0);
      config.staIP[0] = 0; 
      config.staGateway[0] = 0; 
      config.staMask[0] = 0; 
    } else {
      index = 1;
      return etUnknownParam;
    }
  } else if (counter < 4) {
    //sendToLogPf(LOG_DEBUG, PSTR("counter < 4: %d"), counter); 
    if (counter > 1) return etMissingParam;
  } else {
     // test all valid IP addresses
     for (int i = 1; i < 3; i++) {
       if (!ipa.isValid(token[i])) {
        index = i;
        return etInvalidNumber;
       }  
     }
     if ((!restartClient) && strcmp(config.staIP, token[1].c_str())) restartClient = true;
     strlcpy(config.staIP, token[1].c_str(), IP_SZ);     
     if ((!restartClient) && strcmp(config.staGateway, token[2].c_str())) restartClient = true;
     strlcpy(config.staGateway, token[2].c_str(), IP_SZ);     
     if ((!restartClient) && strcmp(config.staMask, token[3].c_str())) restartClient = true;
     strlcpy(config.staMask, token[3].c_str(), IP_SZ);     
  }
  //sendToLogPf(LOG_DEBUG, PSTR("restartClient %s"), (restartClient) ? "yes" : "no");
  if (restartClient) {
    sendToLogP(LOG_INFO, PSTR("Restarting client"));
    setWiFiStationConfig();
    connectWiFi(config.netSsid, config.netPsk);
  }

  char msg[MSG_SZ] = {0};
  if (strlen(config.staIP))
    strlcpy(msg, "Static", MSG_SZ);
  else
    strlcpy(msg, "Dynamic (DHCP)", MSG_SZ);
  sendToLogPf(LOG_INFO, PSTR("%s address, IP: %s, gateway: %s, and subnet mask: %s"), 
    msg, WiFi.localIP().toString().c_str(), WiFi.gatewayIP().toString().c_str(), WiFi.subnetMask().toString().c_str() );
    
  if (counter > 4) {
    index = 4;
    return etExtraParam;
  }
  return etNone;
}  


//      1       2   <<< counter
//      0       1   <<< index
//  "config (save|load|default|erase)"
//
cmndError_t doConfig(int counter, int &index) {
  if (counter < 2) return etMissingParam;
  index = 1;
  token[1].toLowerCase();
  if (token[1].equals("version")) getConfigVersion();
  else if (token[1].equals("save")) saveConfigToEEPROM();
  else if (token[1].equals("load")) loadConfigFromEEPROM();
  else if (token[1].equals("default")) useDefaultConfig();
  else if ( token[1].equals("erase") ) clearEEPROM();
  else return etUnknownParam;
  if (counter > 2) {
    index = 2;
    return etExtraParam;
  }
  return etNone;
}

//      1       2   <<< counter
//      0       1   <<< index
//  "config [<wait>]"
//
cmndError_t doCycle(int counter, int &index) {
  unsigned long len = 0;
  if (counter > 1) {
    len = token[1].toInt();
    if (len < 0) {
      index = 1;
      return etInvalidNumber;
    }
  }    
  startCycling(len);
  if (counter > 2) {
    index = 2;
    return etExtraParam;
  }
  return etNone;
}

//      1       2   <<< counter
//      0       1   <<< index
//  "help   [<command>]"
//
cmndError_t doHelp(int counter, int &index) {
  char space = ' ';
  char msg[MSG_SZ] = {0};
  int cid;
  index = 1;

  if (counter == 1) {
    strncpy(msg, "commands:", MSG_SZ);
    for (int i=0; i < COMMAND_COUNT; i++) {
      if (strlen(msg) < MSG_SZ) strncat(msg, &space, 1);
      strncat(msg, cmds[i], MSG_SZ-strlen(msg));
    }
    sendToLog(LOG_INFO, msg);
  } else {
    cid = commandId(1, counter);
    if (cid < 0) return etUnknownParam;
    sendToLogf(LOG_INFO, "%s %s", cmds[cid], params[cid]);
  } 

  if (counter > 2) {
    index = 2;
    return etExtraParam;
  }
  return etNone;
}

//    1         2             3  <<< counter
//    0         1             2  <<< index
//  "log (uart|mqtt|web) [<level>]"
//
cmndError_t doLog(int counter, int &index) {
  if (counter < 2) return etMissingParam;  
  index = 1;
  int lg;
  int lv = -1;
  if (counter > 1) {
    token[1].toLowerCase();
    if (token[1].equals("uart"))
      lg = 0; 
    else if (token[1].equals("mqtt"))
      lg = 1; 
    else if (token[1].equals("syslog"))
      lg = 2;  
    else
      return etUnknownParam;
    if (counter > 2) {  
      token[2].toLowerCase();
      for (int i = 0; i<=7; i++) {
        if (token[2].equals(logLevelString[i])) {
          lv = i;
          break;
        }
      }
      if ((lv < 0) && (token[2].length()==1)) {
        lv = (byte)token[2][0] - '0';
      }   
      if ( (lv < 0) || (lv > 7) ) {
        index = 2;
        return etInvalidNumber;
      }
    }
  }    
  switch (lg) {    
    case 0: { 
      if (counter > 2) {config.logLevelUart = lv;}
      sendToLogPf(LOG_INFO, PSTR("%s log level: %s"), "uart", logLevelString[config.logLevelUart]);
      break;
    }
    case 1: { 
      if (counter > 2) {config.logLevelMqtt = lv;}
      sendToLogPf(LOG_INFO, PSTR("%s log level: %s"), "mqtt", logLevelString[config.logLevelMqtt]);
      break;
    }
    case 2: { 
      if (counter > 2) {config.logLevelSyslog = lv;}
      sendToLogPf(LOG_INFO, PSTR("%s log level: %s"), "syslog", logLevelString[config.logLevelSyslog]);
      break;
    }
  }
  if (counter > 3) {
    index = 3;
    return etExtraParam;
  }
  return etNone;
}


//    1        2     3  <<< counter
//    0        1     2  <<< index
// "monitor [on|off]"
//
cmndError_t doMonitor(int counter, int &index) {
  index = 1;
  if (counter > 1) {
    token[1].toLowerCase();
    if (token[1].equals("off"))
      disableMonitor(); 
    else if (token[1].equals("on"))
      enableMonitor(); 
    else
      return etUnknownParam;
  }

  int ndx = (monitorState != SPINNING) ? monitorState : (netDownReason == WIFI_DOWN) ? monitorState : monitorState+1;
  sendToLogPf(LOG_INFO, ("%s, connected to Wi-Fi network: %s"), monitorStateString[ndx], (WiFi.status() == WL_CONNECTED) ? "yes" : "no");
  if (counter > 2) {
    index = 2;
    return etExtraParam;
  }
  return etNone;
}


//      1      2     3 <<< counter
//      0      1     2 <<< index
//  "mqtt   [<ip> [<port>]]"
//
cmndError_t doMqtt(int counter, int &index) {
  IPAddress ipa;
  int aPort = 1883;
  if (counter > 1) {
    if (!ipa.isValid(token[1])) {
      index = 1;
      return etInvalidNumber;
    }
    strncpy(config.mqttHost, token[1].c_str(), IP_SZ);
    if (counter > 2) {
      aPort = token[2].toInt();
      if (aPort <= 0) {
        index = 2;
        return etInvalidNumber;
      }
    }  
    config.mqttPort = aPort;
    // restart mqtt client !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  }  
  sendToLogPf(LOG_INFO, PSTR("MQTT host: %s, port: %d, connected: %s"), config.mqttHost, config.mqttPort, (mqttClient.connected()) ? "yes" : "no");
  if (counter > 2) {
    index = 2;
    return etExtraParam;
  }
  return etNone;
}


bool isValidHostnameChar(char c) {
  Serial.printf("isValidHostChar(\"%c\")\n", c);
  if (c == '-') return true;
  if (c < '0')  return false;
  if (c <= '9') return true; 
  if (c < 'A')  return false;
  if (c <= 'Z') return true;
  if (c < 'a')  return false;
  if (c <= 'z') return true;
  return false;
}

//      1       2   <<< counter
//      0       1   <<< index
//  "name   [<hostname>]"
//
// Allowed chars: "a".."z", "A".."Z", "0".."9" "-", but must not start nor end with "-". 
cmndError_t doName(int counter, int &index) {
  if (counter > 1) {
    char newhost[HOST_NAME_SZ] = {0};
    index = 1;
    int len = token[1].length();
    for (int i=0; i < len; i++) {
      if (!isValidHostnameChar(token[1].charAt(i))) {
        return etInvalidNumber;  
      }
    }  
    if ( (token[1].charAt(0) == '-') || (token[1].charAt(len-1) == '-') )
      return etInvalidNumber;  
    strlcpy(config.hostname, token[1].c_str(), HOST_NAME_SZ);
  }
  sendToLogPf(LOG_INFO, PSTR("hostname: %s%s"), config.hostname, (counter > 1) ? " - must save config and restart to take effect" : "");
  if (counter > 2) {
    index = 2;
    return etExtraParam;
  }
  return etNone;
  ///// need to restart
}


//    1      2        3  <<< counter
//    0      1        2  <<< index
//  "net [<ssid> [<password>]]"
//
cmndError_t doNet(int counter, int &index, int source) {
  char pwd[PWD_SZ] = {0};
  char ssid[SSID_SZ] = {0};

  if (counter == 1) 
    sendToLogf(LOG_INFO, "%s> %s", cmdsrc[source], token[0].c_str());
  else {
    sendToLogf(LOG_INFO, "%s> %s %s", cmdsrc[source], token[0].c_str(), token[1].c_str());
    disableMonitor();
    strlcpy(ssid, token[1].c_str(), PWD_SZ);
    token[1].toLowerCase();
    if (token[1].equals("-clear")) {
      WiFi.disconnect();
      delay(100);
    } else {
      if (counter > 2) {
        if (strlen(token[2].c_str()) < 8) {
          sendToLogP(LOG_ERR, PSTR("WiFi password must be at least 8 characters long"));
          index = 2;
          return etUnknownParam;
        }
        strlcpy(pwd, token[2].c_str(), PWD_SZ);
      }
      connectWiFi(ssid, pwd);
    }  
    enableMonitor();  
  }
  if (WiFi.SSID().length()) {
    sendToLogPf(LOG_INFO, PSTR("WiFi SSID: %s, connected: %s"), WiFi.SSID().c_str(), WiFi.isConnected() ? "yes" : "no");
  } else {
    sendToLogP(LOG_INFO, PSTR("WiFi SSID: undefined"));
  }
  if (counter > 3) {
    index = 3;
    return etExtraParam;
  }
  return etNone;
}


//    1      2      2        3  <<< counter
//    0      1      1        2  <<< index
//  "ota [-clear|<ssid> [<password>]]"
//
cmndError_t doOta(int counter, int &index, int source) {
  char ssid[SSID_SZ] = {0};

  if (counter == 1) 
    sendToLogf(LOG_INFO, "%s> %s", cmdsrc[source], token[0].c_str());
  else {   
    sendToLogf(LOG_INFO, "%s> %s %s", cmdsrc[source], token[0].c_str(), token[1].c_str());
    strlcpy(ssid, token[1].c_str(), PWD_SZ);
    token[1].toLowerCase();
  
    if (token[1].equals("default")) {
      config.otaSsid[0] = '\0'; 
      config.otaPsk[0] = '\0';
    } else {
      if (counter > 2) {
        if (strlen(token[2].c_str()) < 8) {
          sendToLogP(LOG_ERR, PSTR("Password must be at least 8 characters long"));
          index = 2;
          return etUnknownParam;
        }
        strlcpy(config.otaPsk, token[2].c_str(), PWD_SZ);
      }  
      if (strcmp(ssid, "\"\""))
        strlcpy(config.otaSsid, ssid, SSID_SZ);
      else      
        config.otaSsid[0] = '\0';
    }
  }
  if (!strlen(config.otaSsid)) {
    if (WiFi.SSID().length()) {
      strlcpy(ssid, WiFi.SSID().c_str( ), SSID_SZ);
    } else {
      strcpy_P(ssid, PSTR("not defined"));
    }
  } else {
    strlcpy(ssid, config.otaSsid, SSID_SZ);
  }
  sendToLogPf(LOG_INFO, PSTR("OTA SSID: %s"), ssid);
  //sendToLogPf(LOG_INFO, PSTR("DBG OTA SSID: \"%s\", PSK: \"%s\""), ssid, config.otaPsk);
       
  if (counter > 3) {
    index = 3;
    return etExtraParam;
  }
  return etNone;
}


//    1      2     3  <<< counter
//    0      1     2  <<< index
//  "ping <host>"
cmndError_t doPing(int counter, int &index) {
  IPAddress ip;
  int i = 2;
  if (counter < 2) {
    return etMissingParam;
  }
  if (counter > 2) { 
    i = 3;
    token[1].toLowerCase();
    if ( (!token[1].equals("-d")) && (!token[1].equals("-dns")) ) {
      index = 1;
      return etUnknownParam;
     }
     if ( WiFi.hostByName(token[2].c_str(), ip, 1000) ) {
       sendToLogPf(LOG_INFO, PSTR("DNS lookup of \"%s\" succeeded, ip: %s"), token[2].c_str(), ip.toString().c_str() );
     } else {
       sendToLogPf(LOG_INFO, PSTR("DNS lookup of \"%s\" failed"), token[2].c_str() );
     }  
  } else {
    int pingResult = pingHost(token[1].c_str(), 1);
    if (pingResult < 0) {
      sendToLogf(LOG_INFO, PSTR("\"%s\" is unreachable"), token[1].c_str());
    } else {
      sendToLogPf(LOG_INFO, PSTR("Pinged \"%s\" in %d ms"), token[1].c_str(), pingResult);
    }
  }  
  if (counter > i) {
    index = i;
    return etExtraParam;
  }
  return etNone;
}


//    1      2        3  <<< counter
//    0      1        2  <<< index
// "reach [(1|2|3) (<host>)]"
//
cmndError_t doReach(int counter, int &index) {
  char buff[MSG_SZ] = {0};    
  int ndx = 0;
  if (counter > 1) {
    ndx = token[1].toInt();
    if ((ndx <= 0) || (ndx >= PING_TARGET_COUNT)){
      index = 1;
      return etInvalidNumber;
    }
    ndx--;
    if (counter < 3) {
      return etMissingParam;
    }
    strlcpy(config.targets[ndx], token[2].c_str(), sizeof(config.targets[0]));
  }  
  strcpy(buff, config.targets[0]);
  for (int i=1; i<PING_TARGET_COUNT; i++) {
    strcat(buff, ", ");
    strcat(buff, config.targets[i]);
  }
  sendToLogPf(LOG_INFO, PSTR("Tries to reach %s"), buff);
  if (counter > 3) {
    index = 3;
    return etExtraParam;
  }
  return etNone;  
}


//    1          2          <<< counter
//    0          1          <<< index
// "router [on|off|toggle]
//
cmndError_t doRouter(int counter, int &index) {
  if (counter > 1) {
    token[1].toLowerCase();
   
    if (token[1].equals("on")) {
      disableMonitor();
      setRouterOn();
      setBlinkyPattern(ROUTER_ON_PATTERN);
      
    } else if (token[1].equals("off")) { 
      disableMonitor();
      setRouterOff();
      setBlinkyPattern(ROUTER_OFF_PATTERN);
      
    } else if (token[1].equals("toggle")) {
      disableMonitor();
      if (toggleRouter()) {
        setBlinkyPattern(ROUTER_ON_PATTERN);
      } else {
        setBlinkyPattern(ROUTER_OFF_PATTERN);  
      }
      
    } else {  
      index = 1;
      return etUnknownParam;
    }
  }  
  //sendToLogPf(LOG_INFO, PSTR("Router: %s. WiFi connected: %s"), isRouterOn() ? "on" : "off", (WiFi.status()==WL_CONNECTED) ? "yes" : "no");
  if (counter > 2) {
    index = 2;
    return etExtraParam;
  }
  return etNone;
}


//      1      2     3 <<< counter
//      0      1     2 <<< index
//  "syslog   [<ip> [<port>]]"
//
cmndError_t doSyslog(int counter, int &index) {
  IPAddress ipa;
  int aPort = 514;
  if (counter > 1) {
    if (!ipa.isValid(token[1])) {
      index = 1;
      return etInvalidNumber;
    }
    strncpy(config.syslogHost, token[1].c_str(), IP_SZ);
    if (counter > 2) {
      aPort = token[2].toInt();
      if (aPort <= 0) {
        index = 2;
        return etInvalidNumber;
      }
    }  
    config.syslogPort = aPort;
  }  
  sendToLogPf(LOG_INFO, PSTR("Syslog host: %s, port: %d"), config.syslogHost, config.syslogPort);
  if (counter > 2) {
    index = 2;
    return etExtraParam;
  }
  return etNone;
}


//   1                               2                                  3     <<< counter
//   0                               1                                  2     <<< index
// "time (wifi|internet|ping|longwait|shortwait|cycling|connect|mqtt) [<ms>])"
//
cmndError_t doTime(int counter, int &index) {
  long aTime;
  
  if (counter < 2) return etMissingParam;

  if (counter > 2) {
    aTime = token[2].toInt();
    if (aTime <= 0) {
      index = 2;
      return etInvalidNumber;
    }
  }  

  token[1].toLowerCase(); 
      
  if (token[1].equals("wifi")) {
    if (counter > 2) { config.wifiDownInterval = aTime; }  
    sendToLogPf(LOG_INFO, PSTR("Wi-Fi down time: %d ms"), config.wifiDownInterval);

  } else if (token[1].equals("internet")) {
    if (counter > 2) { config.internetLostInterval = aTime; }  
    sendToLogPf(LOG_INFO, PSTR("internet lost time: %li ms"), config.internetLostInterval);

  } else if (token[1].equals("ping")) {
    if (counter > 2) { config.intervalBetweenPings = aTime; }  
    sendToLogPf(LOG_INFO, PSTR("time between pings: %li ms"), config.intervalBetweenPings);
      
  } else if (token[1].equals("longwait")) {
    if (counter > 2) { config.waitAfterInternetLost = aTime;}  
    sendToLogPf(LOG_INFO, PSTR("longwait after internet lost: %li ms"), config.waitAfterInternetLost);

  } else if (token[1].equals("shortwait")) {
    if (counter > 2) { config.waitAfterWifiDown = aTime; }  
    sendToLogPf(LOG_INFO, PSTR("shortwait after Wi-Fi downime: %li ms"), config.waitAfterWifiDown);
      
  } else if (token[1].equals("cycling")) {
    if (counter > 2) { config.routerOffInterval = aTime; }  
    sendToLogPf(LOG_INFO, PSTR("router cycling time: %d ms"), config.routerOffInterval);

  } else if (token[1].equals("connect")) {
    if (counter > 2) { config.connectTime = aTime; }  
    sendToLogPf(LOG_INFO, PSTR("Wi-Fi connection timeout: %d ms"), config.routerOffInterval);

  } else if (token[1].equals("mqtt")) {
    if (counter > 2) { config.mqttInterval = aTime; }  
    sendToLogPf(LOG_INFO, PSTR("time beween mqtt reconnect attempts: %d ms"), config.mqttInterval);

  } else {  
      index = 1;
      return etUnknownParam;
  }

  if (counter > 3) {
    index = 3;
    return etExtraParam;
  }
  return etNone;
}

//      1       2       3 <<< counter
//      0       1       2 <<< index
//  "topic  (in|out) [<topic>]"
//
cmndError_t doTopic(int counter, int &index) {
  if (counter < 2) return etMissingParam;

  token[1].toLowerCase(); 
      
  if (token[1].equals("in")) {
    if (counter > 2) { 
      strlcpy(config.mqttCommand, token[2].c_str(), TOPIC_SZ);
    }
    sendToLogPf(LOG_INFO, PSTR("MQTT input topic: %s/%s"), HOST_NAME, config.mqttCommand);

  } else if (token[1].equals("out")) {
    if (counter > 2) { 
      strlcpy(config.mqttResponse, token[2].c_str(), TOPIC_SZ);
    }
    sendToLogPf(LOG_INFO, PSTR("MQTT output topic: %s/%s"), HOST_NAME, config.mqttResponse);
  }
  
  if (counter > 3) {
    index = 3;
    return etExtraParam;
  }
  return etNone;
}

//      1       2   <<< counter
//      0       1   <<< index
//  "update  [<url>]
//
cmndError_t doUpdate(int counter, int &index) {
  char url[MSG_SZ] = {0};

  strlcpy(url, (counter > 1) ? token[1].c_str() : config.otaUrl, MSG_SZ);

  sendToLogPf(LOG_INFO, PSTR("Doing OTA update from %s"), url);
 
  updateOta(url);

  if (counter > 2)  {
    index = 2;
    return etExtraParam;
  }
  return etNone;
}


//    1       2         3    <<< counter
//    0       1         2    <<< index
//  "url  (ota|auto) [<url>]
//
cmndError_t doUrl(int counter, int &index) {
  if (counter < 2) return etMissingParam;

  if (token[1].equals("ota")) {
    if (counter > 2) { strlcpy(config.otaUrl, token[2].c_str(), sizeof(config.otaUrl)); }
    sendToLogPf(LOG_INFO, PSTR("ota ulr \"%s\""), config.otaUrl);

  } else if (token[1].equals("auto")) {
    if (counter > 2) {strlcpy(config.autoUrl, token[2].c_str(), sizeof(config.autoUrl)); }
    sendToLogPf(LOG_INFO, PSTR("auto ulr \"%s\""), config.autoUrl);

  } else {
    index = 1;
    return etUnknownParam;
  }
  if (counter > 3) {
    index = 3;
    return etExtraParam;
  }
  return etNone;
}


//     1       <<< counter
//     0       <<< index
//  "version"
//
cmndError_t doVersion(int counter, int &index) {
  sendToLogPf(LOG_INFO, PSTR("Network Monitor (version %d.%d.%d)"), (VERSION >> 16) & 0xFF, (VERSION >> 8) & 0xFF, VERSION & 0xFF);
  if (counter > 2)  {
    index = 2;
    return etExtraParam;
  }
  return etNone;
}



void doCommand(int source, const char *cmnd) {
  cmndError_t error = etNone;
  int index = 0;
  int n;
  
  int counter = parseString(cmnd);
  if (counter < 1) return;

  n = commandId(0, counter);

  boolean quiet = ((n == NET_COMMAND) || (n == OTA_COMMAND));

  if (!quiet) sendToLogf(LOG_INFO, "%s> %s", cmdsrc[source], cmnd);

  // dbg for (int i = 0; i < counter; i++) { sendToLogf(LOG_DEBUG, "DBG: token[%d]: \"%s\"", i, token[i].c_str()); }
  // dbg sendToLogf(LOG_DEBUG, "DBG: command %d", n);
  switch (n) {
    case  0: error = doClientIp(counter, index); break;
    case  1: error = doConfig(counter, index); break;
    case  2: error = doCycle(counter, index); break;
    case  3: error = doHelp(counter, index); break;
    case  4: error = doLog(counter, index); break;
    case  5: error = doMonitor(counter, index); break;
    case  6: error = doMqtt(counter, index); break;
    case  7: error = doName(counter, index); break;
    case  8: error = doNet(counter, index, source); break;
    case  9: error = doOta(counter, index, source); break;
    case 10: error = doPing(counter, index); break;
    case 11: error = doReach(counter, index); break;
    case 12: doRestart(USER_RESTART_CMD);
    case 13: error = doRouter(counter, index); break;
    case 14: error = doSyslog(counter, index); break;
    case 15: error = doTime(counter, index); break;
    case 16: error = doTopic(counter, index); break;
    case 17: error = doUpdate(counter, index); break;
    case 18: error = doUrl(counter, index); break;
    case 19: error = doVersion(counter, index); break;
    default: error = etUnknownCommand; break;
  }

  switch (error) {
    case etMissingParam:   {
        sendToLogP(LOG_ERR, PSTR("Missing parameter"));
      } break;
    case etUnknownCommand: {
        sendToLogPf(LOG_ERR, PSTR("\"%s\" unknown command"), token[index].c_str());
      } break;
    case etUnknownParam:   {
        sendToLogPf(LOG_ERR, PSTR("\"%s\" unknown parameter"), token[index].c_str());
      } break;
    case etExtraParam:     {
        sendToLogPf(LOG_ERR, PSTR("\"%s\" extra parameter"), token[index].c_str());
      } break;
    case etInvalidNumber:  {
        sendToLogPf(LOG_ERR, PSTR("\"%s\" invalid value"), token[index].c_str());        
    }
    default:               {
        return;
      } break; // etNone
  }
}
