unit TestTokenizer;

interface

uses
  TestFramework, SysUtils, Classes, SimpleTokenizer;

type
  // Mthodes de test pour la classe TMdTokenizer

  // Test TSimpleTokenizer.NextTokenType
  TestNextTokenType = class(TTestCase)
  strict private
    FTokenizer: TMdTokenizer;
    FSource: string;
    FSources: TStrings;
  protected
    procedure SetSource;
    procedure TestSource;
  public
    procedure SetUp; override;
    procedure TearDown; override;
  published
    procedure TestCreate;
    procedure TestWhiteChar;   // ' '
    procedure TestWhiteString; // '  '
    procedure TestDigit;       // '0' to '9';
    procedure TestWhiteDigit;  // ' 0 ' to ' 9 ';
    procedure TestDoubleDigit; // '10' to '19';
    procedure TestDecimals;    // '10.0' to '10.9';
    procedure TestAllOps;      // '-+/*()';
    procedure TestExpression;  // '5+8*(7-2)'
    procedure TestEmptyLine1;  // CR
    procedure TestEmptyLine2;  // CR+LF
    procedure TestEmptyLine3;  // '3+'+CR+LF+'5';
  end;

  // Test TSimpleTokenizer.NextToken
  TestNextToken = class(TTestCase)
  strict private
    FTokenizer: TMdTokenizer;
    FSource: string;
    FSources: TStrings;
  protected
    procedure SetSource;
    procedure TestSource;
  public
    procedure SetUp; override;
    procedure TearDown; override;
  published
    procedure TestCreate;
    procedure TestWhiteChar;   // ' '
    procedure TestWhiteString; // '  '
    procedure TestDigit;       // '0' to '9';
    procedure TestWhiteDigit;  // ' 0 ' to ' 9 ';
    procedure TestDoubleDigit; // '10' to '19';
    procedure TestDecimals;    // '10.0' to '10.9';
    procedure TestAllOps;      // '-+/*()';
    procedure TestExpression;  // '5+8*(7-2)'
    procedure TestEmptyLine1;  // CR
    procedure TestEmptyLine2;  // CR+LF
    procedure TestEmptyLine3;  // '3 + '+CR+LF+' 5';
  end;

implementation

type
  THackToknzr = class(TMdTokenizer);

procedure TestNextTokenType.SetUp;
begin
  FTokenizer := TMdTokenizer.Create;
  FSource := '';
  FSources := TStringList.Create;
end;

procedure TestNextTokenType.TearDown;
begin
  FTokenizer.Free;
  FTokenizer := nil;
  FSource := '';
  FSources.Free;
  FSources := nil;
end;

procedure TestNextTokenType.SetSource;
var
  i: integer;
begin
  FSource := '';
  for I := 0 to FSources.Count - 1 do
    FSource := FSource + FSources[i];
  FTokenizer.Source := FSource;
end;

procedure TestNextTokenType.TestSource;
var
  i: integer;
  p: integer;
  src: string;
begin
  Check(FTokenizer <> nil, 'FTokenizer broken!');
  CheckEquals(FSource, FTokenizer.Source, 'FTokenizer source');
  p := 1;
  for I := 0 to FSources.Count - 1 do begin
    FTokenizer.NextToken;
    CheckEquals(integer(FSources.Objects[i]), ord(FTokenizer.TokenType),
      Format('NextTokenType token="%s" index=%d', [FTokenizer.Token, i]));
    CheckEquals(FSources[i], FTokenizer.Token,
      Format('Token index=%d', [i]));
    CheckEquals(p, FTokenizer.TokenStart,
      Format('TokenStart index=%d', [i]));
    CheckEquals(length(FSources[i]), FTokenizer.TokenLength,
      Format('TokenLength index=%d', [i]));
    p := p + length(FSources[i]);
  end;
  CheckEquals(ord(ttEOS), ord(FTokenizer.NextToken), 'eos');
  CheckEquals(0, FTokenizer.TokenLength, 'eos Tokenlength');
  CheckEquals(p, FTokenizer.TokenStart, 'eos Token start');

  THackToknzr(FTokenizer).Reset;
  src := '';
  while FTokenizer.NextToken <> ttEOS do
    src := src + FTokenizer.Token;

  CheckEquals(FSource, src, 'reconstructing source');
end;

procedure TestNextTokenType.TestWhiteChar;
begin
  FSources.AddObject(' ', pointer(ttWhite));
  SetSource;
  TestSource;
end;

procedure TestNextTokenType.TestWhiteString;
begin
  FSources.Clear;
  FSources.AddObject('  ', pointer(ttWhite));
  SetSource;
  TestSource;
end;

procedure TestNextTokenType.TestCreate;
begin
  SetSource;
  TestSource;
end;

procedure TestNextTokenType.TestDigit;
var
  c: char;
begin
  for c := '0' to '9' do begin
    FSources.Clear;
    FSources.AddObject(c, pointer(ttNumber));
    SetSource;
    TestSource;
  end;
end;

procedure TestNextTokenType.TestWhiteDigit;
var
  c: char;
begin
  for c := '0' to '9' do begin
    FSources.Clear;
    FSources.AddObject(' ', pointer(ttWhite));
    FSources.AddObject(c, pointer(ttNumber));
    FSources.AddObject(' ', pointer(ttWhite));
    SetSource;
    TestSource;
  end;
end;

procedure TestNextTokenType.TestDoubleDigit;
var
  c: char;
begin
  for c := '0' to '9' do begin
    FSources.Clear;
    FSources.AddObject('1' + c, pointer(ttNumber));
    SetSource;
    TestSource;
  end;
end;

procedure TestNextTokenType.TestEmptyLine1;
begin
  FSources.AddObject(#13, pointer(ttWhite));
  SetSource;
  TestSource;
end;

procedure TestNextTokenType.TestEmptyLine2;
begin
  FSources.AddObject(#13#10, pointer(ttWhite));
  SetSource;
  TestSource;
end;

procedure TestNextTokenType.TestEmptyLine3;
begin
  FSources.AddObject('3', pointer(ttNumber));
  FSources.AddObject('+', pointer(ttPlus));
  FSources.AddObject(#13#10, pointer(ttWhite));
  FSources.AddObject('5', pointer(ttNumber));
  SetSource;
  TestSource;
end;

procedure TestNextTokenType.TestExpression;
begin
  // '5+8*(7-2)'
  FSources.AddObject('5', pointer(ttNumber));
  FSources.AddObject('+', pointer(ttPlus));
  FSources.AddObject('8', pointer(ttNumber));
  FSources.AddObject('*', pointer(ttMult));
  FSources.AddObject('(', pointer(ttLeftParenthesis));
  FSources.AddObject('7', pointer(ttNumber));
  FSources.AddObject('-', pointer(ttMinus));
  FSources.AddObject('2', pointer(ttNumber));
  FSources.AddObject(')', pointer(ttRightParenthesis));
  SetSource;
  TestSource;
end;

procedure TestNextTokenType.TestDecimals;
var
  c: char;
begin
  for c := '0' to '9' do begin
    FSources.Clear;
    FSources.AddObject('10' + FTokenizer.DecimalChar + c, pointer(ttNumber));
    SetSource;
    TestSource;
  end;
end;

procedure TestNextTokenType.TestAllOps;
begin
  FSources.AddObject('-', pointer(ttMinus));
  FSources.AddObject('+', pointer(ttPlus));
  FSources.AddObject('*', pointer(ttMult));
  FSources.AddObject('/', pointer(ttDiv));
  FSources.AddObject('(', pointer(ttLeftParenthesis));
  FSources.AddObject(')', pointer(ttRightParenthesis));
  SetSource;
  TestSource;
end;

{ TestNextToken }

procedure TestNextToken.SetUp;
begin
  FTokenizer := TMdTokenizer.Create;
  FSource := '';
  FSources := TStringList.Create;
end;

procedure TestNextToken.TearDown;
begin
  FTokenizer.Free;
  FTokenizer := nil;
  FSource := '';
  FSources.Free;
  FSources := nil;
end;

procedure TestNextToken.SetSource;
var
  i: integer;
begin
  FSource := '';
  for I := 0 to FSources.Count - 1 do
    FSource := FSource + FSources[i];
  FTokenizer.Source := FSource;
end;

procedure TestNextToken.TestSource;
var
  i: integer;
  p: integer;
  ldw: integer;
  src: string;
begin
  Check(FTokenizer <> nil, 'FTokenizer broken!');
  CheckEquals(FSource, FTokenizer.Source, 'FTokenizer source');
  p := 1;
  for I := 0 to FSources.Count - 1 do begin
    src := FSources[i];
    ldw := 0;
    while (src <> '') and (src[1] <= ' ') do begin
      inc(ldw);
      delete(src, 1, 1);
    end;
    if src <> '' then begin
      FTokenizer.NextTrueToken;
      CheckEquals(integer(FSources.Objects[i]), ord(FTokenizer.TokenType),
        Format('NextToken token="%s" index=%d', [FTokenizer.Token, i]));
      CheckEquals(trim(src), FTokenizer.Token,
        Format('Token index=%d', [i]));
      CheckEquals(p+ldw, FTokenizer.TokenStart,
        Format('TokenStart index=%d', [i]));
      CheckEquals(length(trim(src)), FTokenizer.TokenLength,
        Format('TokenLength index=%d', [i]));
    end;
    p := p + length(FSources[i]);
  end;
  CheckEquals(ord(ttEOS), ord(FTokenizer.NextTrueToken), 'eos');
  CheckEquals(0, FTokenizer.TokenLength, 'eos Tokenlength');
  CheckEquals(p, FTokenizer.TokenStart, 'eos Token start');
end;


procedure TestNextToken.TestWhiteChar;
begin
  FSources.AddObject(' ', pointer(ttEos));
  SetSource;
  TestSource;
end;

procedure TestNextToken.TestWhiteString;
begin
  FSources.Clear;
  FSources.AddObject('  ', pointer(ttEos));
  SetSource;
  TestSource;
end;

procedure TestNextToken.TestCreate;
begin
  SetSource;
  TestSource;
end;

procedure TestNextToken.TestDigit;
var
  c: char;
begin
  for c := '0' to '9' do begin
    FSources.Clear;
    FSources.AddObject(c, pointer(ttNumber));
    SetSource;
    TestSource;
  end;
end;

procedure TestNextToken.TestWhiteDigit;
var
  c: char;
begin
  for c := '0' to '9' do begin
    FSources.Clear;
    FSources.AddObject(' ' + c + ' ', pointer(ttNumber));
    SetSource;
    TestSource;
  end;
end;

procedure TestNextToken.TestDoubleDigit;
var
  c: char;
begin
  for c := '0' to '9' do begin
    FSources.Clear;
    FSources.AddObject('  1' + c, pointer(ttNumber));
    SetSource;
    TestSource;
  end;
end;


procedure TestNextToken.TestEmptyLine1;
begin
  FSources.AddObject(#13, pointer(ttEOS));
  SetSource;
  TestSource;
end;

procedure TestNextToken.TestEmptyLine2;
begin
  FSources.AddObject(#13#10, pointer(ttEOS));
  SetSource;
  TestSource;
end;

procedure TestNextToken.TestEmptyLine3;
begin
  FSources.AddObject('3', pointer(ttNumber));
  FSources.AddObject(' +', pointer(ttPlus));
  FSources.AddObject(#13#10' 5', pointer(ttNumber));
  SetSource;
  TestSource;
end;

procedure TestNextToken.TestExpression;
begin
  // '5+8*(7-2)'
  FSources.AddObject(' 5  ', pointer(ttNumber));
  FSources.AddObject('+', pointer(ttPlus));
  FSources.AddObject('8 ', pointer(ttNumber));
  FSources.AddObject(' *', pointer(ttMult));
  FSources.AddObject('( ', pointer(ttLeftParenthesis));
  FSources.AddObject('7', pointer(ttNumber));
  FSources.AddObject(' - ', pointer(ttMinus));
  FSources.AddObject('2', pointer(ttNumber));
  FSources.AddObject('  )', pointer(ttRightParenthesis));
  SetSource;
  TestSource;
end;

procedure TestNextToken.TestDecimals;
var
  c: char;
begin
  for c := '0' to '9' do begin
    FSources.Clear;
    FSources.AddObject('  10' + FTokenizer.DecimalChar + c, pointer(ttNumber));
    SetSource;
    TestSource;
  end;
end;

procedure TestNextToken.TestAllOps;
begin
  FSources.AddObject('-', pointer(ttMinus));
  FSources.AddObject('+', pointer(ttPlus));
  FSources.AddObject('*', pointer(ttMult));
  FSources.AddObject('/', pointer(ttDiv));
  FSources.AddObject('(', pointer(ttLeftParenthesis));
  FSources.AddObject(')', pointer(ttRightParenthesis));
  SetSource;
  TestSource;
end;


initialization
  // Recenser tous les cas de test avec le lanceur de test
  RegisterTest(TestNextTokenType.Suite);
  RegisterTest(TestNextToken.Suite);
end.

