(***********************************************************************
 *
 * SimpleMathParser.pas version 1.0
 *
 * Copyright (c) 2013 Michel Deslierres
 * All rights reserved. Tous droits rservs.
 *
 * Read full license in the file license.txt
 * Lire la licence au complet dans le fichier license.txt
 *
 ************************************************************************)

(*:
   Unit that performs lexical analysis of simple mathematical
   expressions and evaluates them. This is accomplished in a
   one pass recursive scanner.

   The only supported operators and their precedence are

     op   operation       precedence
     --   --------------  ----------
     ()   parenthesis        4
	 -    negation           3
     *    multiplication     2
     /    division           2
     +    addition           1
     -    subtraction        1

WSN definitions of the supported syntax 

 Recall: [ a ]    means a is optional (i.e. 0 or 1 repetition of a)
         { a }    means 0 or more repetitions of a
         a | b    means a or b
		 use ( )  to group elements

DecSym  = ? locale dependant character, usually "." in English, "," in French... ?

Digit      = "0" | ... | "9".
Number     = Digit{Digit}, [ DecSym {Digit} [ ("E" | "e") ["-" | "+"] Digit{Digit } ]].
Term       = Exponent { ("*" | "/" | ":" | "%") Exponent }.
Expression = Term { ("-" | "+") Term }.
Factor     = Number | "(" Expression  ")" | ("-" | "+") Factor.
*)
unit SimpleMathParser;

{$IFNDEF PARSER_INC}
  {$I Parser.inc}
{$ENDIF}

interface

uses
  {$IFDEF DUMP} Classes, {$ENDIF}
  SysUtils, ParserTypes;

type
  {: Parser exception }
  EMathParserErr = class(Exception)
  public
    ParserError: TParserError;
    TokenStart: integer;
    TokenLength: integer;
  end;

{: The function that performs a lexical analysis of a mathematical expression
   and returns the result of the calculation. Can raise an exception of type
   EMathParseErr or EMathError (EInvalidOp, EZeroDivide, EOverflow and
   EUnderflow).}
function EvaluateExpression(const source: string): float;

{: Character used as decimal symbol. By default this is defined in the
   computer's locale (usually a period for English language systems
   and a comma for French language systems).}
function ParserDecimalChar: char;

{: Sets the character to be used as the decimal symbol. Returns the
   previous decimal symbol.}
function SetParserDecimalChar(value: char): char;

{$IFDEF DUMP}
var
  jlevel: integer;
  journal: TStrings;
{$ENDIF}

{$IFDEF DUMP}
const
  SVersionName = 'SimpleMathParser';
  SVersion = '1.0';
  VersionMajor = 1;
  VersionMinor = 0;
{$ENDIF}

implementation

uses
  ParserConsts, SimpleTokenizer;

{ local tokenizer }

var
  tokenizer: TMdTokenizer;

procedure RaiseParseException(Error: TParserError);
var
  E: EMathParserErr;
begin
  E := EMathParserErr.Create(_(SParseError));
  with E do begin
    TokenStart := Tokenizer.TokenStart;
    TokenLength := Tokenizer.TokenLength;
    ParserError := Error;
  end;
  Raise E;
end;

function StrToNumber(const value: string): float;
var
  p: integer;
  s: string;
begin
  {We will be lazy and use StrToFloat to convert
   the string to a number. However StrToFloat
   uses the locale's decimal symbol and
   furthermore StrToFloat('2E') returns 2E1 which
   is too ambiguous. So if the string ends in E
   an exception will be raised.}
  s := value;
  if upcase(s[length(s)]) <> 'E' then begin
    if ParserDecimalchar <> DecimalSeparator then begin
      p := pos(ParserDecimalChar, s);
      while p > 0 do begin
        s[p] := DecimalSeparator;
        p := pos(ParserDecimalChar, s);
      end;
    end;
    try
      result := StrToFloat(s);
      exit;
    except
      // fall through on E: EConvertError
    end;
  end;
  RaiseParseException(peInvalidNumber);
end;

function Factor: float; forward;

function Term: float;
begin
  {$IFDEF DUMP}
  inc(jlevel, 2);
  journal.Add(Format('%*.sTerm entered (token="%s")', [jlevel, ' ', Tokenizer.Token]));
  {$ENDIF}
  result := Factor;
  repeat
    if Tokenizer.TokenType = ttMult then begin
      {$IFDEF DUMP}
      journal.Add(Format('%*.sTerm found "*" will get second factor', [jlevel, ' ']));
      {$ENDIF}
      Tokenizer.NextTrueToken;
      result := result * Factor;
    end
    else if Tokenizer.TokenType = ttDiv then begin
      {$IFDEF DUMP}
      journal.Add(Format('%*.sTerm found "/" will get second factor', [jlevel, ' ']));
      {$ENDIF}
      Tokenizer.NextTrueToken;
      result := result / Factor;
    end
    else begin
      {$IFDEF DUMP}
      journal.Add(Format('%*.sTerm done, value=%g', [jlevel, ' ', result]));
      journal.Add(Format('%*.sTerm done, token="%s"', [jlevel, ' ', Tokenizer.Token]));
      dec(jlevel, 2);
      {$ENDIF}
      exit;
    end;
  until false;
end;

function Expression: float;
begin
  {$IFDEF DUMP}
  inc(jlevel, 2);
  journal.Add(Format('%*.sExpression entered (token="%s")', [jlevel, ' ', Tokenizer.Token]));
  {$ENDIF}

  result := Term;
  repeat
    if Tokenizer.TokenType = ttPlus then begin
      {$IFDEF DUMP}
      journal.Add(Format('%*.sExpression found "+" will get second term', [jlevel, ' ']));
      {$ENDIF}
      Tokenizer.NextTrueToken;
      result := result + Term;
    end
    else if Tokenizer.TokenType = ttMinus then begin
      {$IFDEF DUMP}
      journal.Add(Format('%*.sExpression found "-" will get second term', [jlevel, ' ']));
      {$ENDIF}
      Tokenizer.NextTrueToken;
      result := result - Term;
    end
    else begin
      {$IFDEF DUMP}
      journal.Add(Format('%*.sExpression done, value=%g', [jlevel, ' ', result]));
      journal.Add(Format('%*.sExpression done, token="%s"', [jlevel, ' ', Tokenizer.Token]));
      dec(jlevel, 2);
      {$ENDIF}
      exit;
    end;
  until false;
end;

function Factor: float;
begin
  {$IFDEF DUMP}
  inc(jlevel, 2);
  journal.Add(Format('%*.sFactor entered (token="%s")', [jlevel, ' ', Tokenizer.Token]));
  {$ENDIF}

  result := 0;
  if Tokenizer.TokenType = ttNumber then begin
    result := StrToNumber(Tokenizer.Token);
    {$IFDEF DUMP}
    journal.Add(Format('%*.sFactor found number (value=%g)', [jlevel, ' ', result]));
    {$ENDIF}
    Tokenizer.NextTrueToken;
  end
  else if Tokenizer.TokenType = ttPlus then begin
    Tokenizer.NextTrueToken;  // skip and ignore leading '+'
    result := Factor;
    {$IFDEF DUMP}
    journal.Add(Format('%*.sFactor found unary + (value=%g)', [jlevel, ' ', result]));
    {$ENDIF}
  end
  else if Tokenizer.TokenType = ttMinus then begin
    Tokenizer.NextTrueToken;  // skip '-'
    result := - Factor;       // unary -
    {$IFDEF DUMP}
    journal.Add(Format('%*.sFactor found unary - (value=%g)', [jlevel, ' ', result]));
    {$ENDIF}
  end
  else if Tokenizer.TokenType = ttLeftParenthesis then begin
    Tokenizer.NextTrueToken;  // skip '('
    result := Expression;
    {$IFDEF DUMP}
    journal.Add(Format('%*.sFactor found ( + (value=%g)', [jlevel, ' ', result]));
    {$ENDIF}
    if Tokenizer.TokenType <> ttRightParenthesis then begin
      RaiseParseException(peExpectedRightPar);
    end;
    Tokenizer.NextTrueToken; // skip ')'
  end
  else if Tokenizer.TokenType = ttEOS then
    RaiseParseException(peUnexpectedEOS)
  else
    RaiseParseException(peUnexpectedElement);

  {$IFDEF DUMP}
  journal.Add(Format('%*.sFactor done, value=%g', [jlevel, ' ', result]));
  journal.Add(Format('%*.sFactor done, token="%s"', [jlevel, ' ', Tokenizer.Token]));
  dec(jlevel, 2);
  {$ENDIF}
end;

function EvaluateExpression(const source: string): float;
begin
  {$IFDEF DUMP}
  journal.Clear;
  journal.add('Evaluate "' + source + '"');
  {$ENDIF}
  tokenizer.Source := source;
  tokenizer.NextTrueToken; // prime the pump
  result := Expression;
  if (Tokenizer.TokenType <> ttEOS) then
    RaiseParseException(peExpectedEOS);
end;

function ParserDecimalChar: char;
begin
  result := tokenizer.DecimalChar;
end;

function SetParserDecimalChar(value: char): char;
begin
  result := tokenizer.DecimalChar;
  tokenizer.DecimalChar := value;
end;

initialization
  {$IFDEF DUMP}
  jlevel := 0;
  journal := TStringList.Create;
  {$ENDIF}
  tokenizer := TMdTokenizer.create;
finalization
  tokenizer.free;
  {$IFDEF DUMP}
  journal.Free;
  {$ENDIF}
end.
