/*
 * ChipInfo
 * Display some information about the ESP32 system
 *
 *
 * Copyright (C) 2021 Michel Deslierres
 */
/* SPDX-License-Identifier: 0BSD */
/* SPDX-URL: https://spdx.org/licenses/0BSD.html */


#include <Arduino.h>

void printFactoryMac(void) {
  union {
    uint64_t factmac;
    char bytes[sizeof(factmac)];
   } mac;

  Serial.print("  MAC: ");
  mac.factmac = ESP.getEfuseMac();
  for (int i=0; i<6; i++) {
    Serial.printf("%02x", mac.bytes[i]);
    if (i<5) Serial.print(":"); else Serial.println();
  }
}

void printFlashChipMode(FlashMode_t mode) {
  switch (mode) {
    case FM_QIO:  Serial.print("FM_QIO"); break;
    case FM_QOUT: Serial.print("FMQOUT"); break;
    case FM_DIO:  Serial.print("FM_DIO"); break;
    case FM_DOUT: Serial.print("FM_DOUT"); break;
    case FM_FAST_READ: Serial.print("FM_FAST_READ"); break;
    case FM_SLOW_READ: Serial.print("FM_SLOW_READ"); break;
	default: Serial.print("FM_UNKNOWN");
  }
}

void getInfo(void) {
  Serial.println("\n\nESP32 Chip Information\n");
	Serial.printf("Chip model: %s, Revision: %d\n", ESP.getChipModel(), ESP.getChipRevision());
	Serial.printf("  Core count: %d \n", ESP.getChipCores());
	Serial.printf("  CPU frequency: %d MHz\n", ESP.getCpuFreqMHz());
	Serial.printf("  Cycle count: %d\n", ESP.getCycleCount());
  printFactoryMac();
	Serial.printf("  SDK version: %s\n", ESP.getSdkVersion());

	Serial.println("\nInternalRam");
	Serial.printf("  Heap size: %d\n", ESP.getHeapSize()); //total heap size
	Serial.printf("  Free heap: %d\n", ESP.getFreeHeap()); //available heap
	Serial.printf("  Min free heap: %d\n", ESP.getMinFreeHeap()); //lowest level of free heap since boot
	Serial.printf("  Max heap alloc size: %d\n", ESP.getMaxAllocHeap()); //largest block of heap that can be allocated

    Serial.println("\nSketchSize");
    Serial.printf("  Sketch size: %d\n", ESP.getSketchSize());
	Serial.printf("  Free sketch space: %d\n", ESP.getFreeSketchSpace());

	Serial.println("\nFlash Memory");
	Serial.printf("  Flash size: %d\n", ESP.getFlashChipSize());
	Serial.printf("  Flash speed: %d\n", ESP.getFlashChipSpeed());
	Serial.print("  Flash mode: ");
	printFlashChipMode(ESP.getFlashChipMode());
	Serial.printf(" (%d)\n", ESP.getFlashChipMode());

	Serial.println("\nSPI RAM Memory (PSRAM)");
	Serial.printf("  PSRAM size: %d\n", ESP.getPsramSize());
	Serial.printf("  Free PSRAM: %d\n", ESP.getFreePsram());
	Serial.printf("  Min free PSRAM: %d\n", ESP.getMinFreePsram());
	Serial.printf("  Max PSRAM alloc size: %d\n", ESP.getMaxAllocPsram());
}

void setup() {
	Serial.begin(BAUD);
	delay(100);
	getInfo();
}

void loop() {
  // do nothing!
}
